# -*- coding: utf-8 -*-
import abc
import collections
import inspect
import logging
import os
import re
import signal
import sys
from collections import defaultdict
import threading

import sandbox.sdk2 as sdk2
import six

from sandbox.projects.rope import parameters
from sandbox.projects.rope.utils import cached_property

ARGUMENTS_PARAM_NAME = 'arguments'
EXECUTABLE_PARAM_NAME = 'executable'

LEGACY_TMP_NAME = ['_task_type', '_task_executable']


tmp_name_ptrn = re.compile(r'^_[^_]*$')

rope_logger = logging.getLogger('rope')


class AbstractExecutor(six.with_metaclass(abc.ABCMeta, object)):
    @abc.abstractmethod
    def run(self):
        return


class BinaryResourceExecutor(AbstractExecutor):
    def __init__(self, task_executable_resource, task_params, sdk2_sb_obj, logger=None):
        """
        :type task_executable_resource: sdk2.Resource
        :type task_params: parameters.TaskParams | dict[str, str]
        """
        self.task_executable_resource = task_executable_resource
        self.task_params = task_params
        self.sdk2_sb_obj = sdk2_sb_obj
        self.logger = logger or rope_logger

    def run(self):
        task_executable_data = sdk2.ResourceData(self.task_executable_resource)
        task_executable_data_path = str(task_executable_data.path)
        if isinstance(self.task_params, dict):
            task_env_ars = self.task_params
        elif isinstance(self.task_params, parameters.TaskParams):
            task_env_ars = self.task_params.to_env_args()
        else:
            raise ValueError('task_param has unsupported type {}'.format(type(self.task_params)))

        # TODO: remove after debugging
        self.logger.info(
            'task_executable_data={}'
            ' task_executable_data_path={}'
            ' task_executable_resource={}'
            ' task_params_class={}'
            ' env={}'
            ''.format(
                task_executable_data,
                task_executable_data_path,
                self.task_executable_resource.type,
                type(self.task_params),
                task_env_ars,
            )
        )

        with sdk2.helpers.ProcessLog(self.sdk2_sb_obj, logger=self.logger) as pl:
            process = sdk2.helpers.subprocess.Popen(
                [task_executable_data_path],
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=task_env_ars,
            )

            self.sdk2_sb_obj.Context.executor_pid = process.pid
            self.sdk2_sb_obj.Context.save()

            process.wait()

    @classmethod
    def on_stop(cls, sdk2_sb_obj, logger=None):
        logger = logger or logging
        executor_pid = getattr(sdk2_sb_obj.Context, 'executor_pid', None)
        if executor_pid:
            try:
                os.kill(executor_pid, signal.SIGTERM)
            except OSError as e:
                logger.exception('Failed to send SIGTERM to process %d, error %s', executor_pid, e)
            else:
                logger.info('Sent SIGTERM to process %d', executor_pid)


def rm_tmp_vars_from_frame_locals(names):
    frame = inspect.currentframe().f_back
    for name in names:
        if name not in LEGACY_TMP_NAME and tmp_name_ptrn.match(name):
            del frame.f_locals[name]


def get_resource_task_name(res):
    if hasattr(res, 'task_name'):
        return res.task_name
    return str(res).lower()


class TaskInitializer(object):

    @property
    @abc.abstractmethod
    def task_key(self):
        return ''

    def as_kv_tuple(self):
        return self.task_key, self

    @property
    @abc.abstractmethod
    def task_name(self):
        return ''

    @property
    @abc.abstractmethod
    def task_prefix(self):
        return ''

    @property
    def param_name_for_executable_arguments(self):
        return self.task_prefix + ARGUMENTS_PARAM_NAME

    @property
    @abc.abstractmethod
    def task_params_class(self):
        """
        :rtype: type[parameters.TaskParams] | parameters.TaskParams | None
        """
        return None

    def get_task_params(self, sdk2_params, sdk2_context):
        if self.task_params_class:
            return self.task_params_class.from_sdk2_params(
                sdk2_params,
                prefix=self.task_prefix,
                context=sdk2_context,
            )

        if hasattr(sdk2_params, self.param_name_for_executable_arguments):
            rope_logger.info('Getting task_arguments from Parameters.%s',
                             self.param_name_for_executable_arguments)
            return getattr(sdk2_params, self.param_name_for_executable_arguments)
        raise ValueError('There is neither task_arguments nor {}'.format(
            self.param_name_for_executable_arguments))

    @abc.abstractmethod
    def get_executor(self, sdk2_sb_obj, logger=None):
        """
        :type sdk2_sb_obj: sdk2.Task
        :type logger: logging.Logger
        :rtype: AbstractExecutor
        """
        return None

    @classmethod
    def on_stop(cls, sdk2_sb_obj, logger=None):
        pass

    @classmethod
    def create_from(cls, create_args):
        if isinstance(create_args, TaskInitializer):
            return create_args
        original_create_args = create_args
        task_params_class = None
        if isinstance(create_args, tuple):
            if len(create_args) == 2:
                create_args, task_params_class = create_args

        if isinstance(create_args, type) and issubclass(create_args, sdk2.Resource):
            return BinaryResourceTaskInitializer(create_args, task_params_class)

        raise ValueError('Unsupported args to create RunnerTask from {}'.format(original_create_args))

    def iter_sdk2_executor_init_param_data(self):
        return
        yield  # noqa

    def iter_sdk2_executor_param_data(self):
        for param_data in self.iter_sdk2_executor_init_param_data():
            yield param_data
        if self.task_params_class:
            for param_data in self.task_params_class.iter_sdk2_task_specific_param_data():
                yield param_data
        else:
            yield parameters.Sdk2ParamCreationData(
                ARGUMENTS_PARAM_NAME,
                param_class=sdk2.parameters.Dict,
                param_kwargs=dict(label='Task arguments'),
            )


class FunctionTaskInitializer(TaskInitializer):

    def __init__(self, task_key, task_params_class, func, name=None, prefix_name=None, on_stop=None):
        self._task_key = task_key
        self._func = func
        self._task_param_class = task_params_class
        self._name = name
        self._prefix_name = prefix_name
        self._on_stop = on_stop

    @property
    def task_key(self):
        return self._task_key

    @property
    def task_name(self):
        return self._name or self.task_key

    class SimpleExecutor(AbstractExecutor):
        def __init__(self, func, task_params, sdk2_sb_obj):
            self.func = func
            self.task_params = task_params
            self.sdk2_sb_obj = sdk2_sb_obj

        def run(self):
            return self.func(self.task_params, self.sdk2_sb_obj)

    @property
    def task_prefix_name(self):
        return self._prefix_name or self.task_key

    @property
    def task_prefix(self):
        return self.task_prefix_name + '_'

    @property
    def task_params_class(self):
        return self._task_param_class

    def get_executor(self, sdk2_sb_obj, logger=None):
        return self.SimpleExecutor(
            func=self._func,
            task_params=self.get_task_params(sdk2_sb_obj.Parameters, sdk2_sb_obj.Context),
            sdk2_sb_obj=sdk2_sb_obj
        )

    def on_stop(self, sdk2_sb_obj, logger=None):
        if self._on_stop:
            task_params = self.get_task_params(sdk2_sb_obj.Parameters, sdk2_sb_obj.Context)
            self._on_stop(task_params, sdk2_sb_obj)


class BinaryResourceTaskInitializer(TaskInitializer):

    def __init__(self, task_executable_resource_type, task_params_class, binary_res_executor_class=None):
        self.task_executable_resource_type = task_executable_resource_type
        self._task_param_class = task_params_class
        self._binary_res_executor_class = binary_res_executor_class or BinaryResourceExecutor

    @property
    def task_key(self):
        return str(self.task_executable_resource_type)

    @property
    def task_name(self):
        return get_resource_task_name(self.task_executable_resource_type)

    @property
    def task_params_class(self):
        return self._task_param_class

    @property
    def task_prefix(self):
        return self.task_name + '_'

    @property
    def param_name_for_executable_resource_type(self):
        return self.task_prefix + EXECUTABLE_PARAM_NAME

    def iter_sdk2_executor_init_param_data(self, base_class=None):
        if base_class and hasattr(base_class, '_task_executable'):
            return
        yield parameters.Sdk2ParamCreationData(
            EXECUTABLE_PARAM_NAME, param_class=sdk2.parameters.Resource,
            param_kwargs=dict(
                label='Task executable',
                resource_type=self.task_executable_resource_type,
                required=True,
            ),
        )

    def get_task_params(self, sdk2_params, sdk2_context):
        if self.task_params_class:
            return self.task_params_class.from_sdk2_params(
                sdk2_params,
                prefix=self.task_prefix,
                context=sdk2_context,
            )

        param_name = self.param_name_for_executable_arguments
        if hasattr(sdk2_params, param_name):
            rope_logger.info('Getting task_arguments from Parameters.%s', param_name)
            return getattr(sdk2_params, param_name)
        raise ValueError('There is neither task_arguments nor {}'.format(param_name))

    def get_executor(self, sdk2_sb_obj, logger=None):
        if hasattr(sdk2_sb_obj.Parameters, '_task_executable'):
            rope_logger.info('Getting _task_executable from Parameters._task_executable')
            task_executable_resource = sdk2_params._task_executable  # noqa
        else:
            param_name = self.param_name_for_executable_resource_type
            rope_logger.info('Getting _task_executable from Parameters.%s', param_name)
            task_executable_resource = getattr(sdk2_sb_obj.Parameters, param_name)

        return self._binary_res_executor_class(
            task_executable_resource,
            task_params=self.get_task_params(sdk2_sb_obj.Parameters, sdk2_sb_obj.Context),
            sdk2_sb_obj=sdk2_sb_obj,
            logger=logger,
        )

    def on_stop(self, sdk2_sb_obj, logger=None):
        self._binary_res_executor_class.on_stop(sdk2_sb_obj, logger=logger)


def get_run_params(task_initializer_or_params_class_or_resource_or_list, task_params_class=None,
                   base_class=sdk2.Parameters):
    """
    Create Parameters for run executable resource task
    or just transform parameters.TaskParams to sdk2.Parameters
    Example:
        from __future__ import print_function
        import sandbox.sdk2 as sdk2
        from sandbox.projects.inventori.common import parameters
        from sandbox.projects.inventori.RunTaskTemplate import RunTaskTemplate, get_run_params

        class TaskFooRes(sdk2.Resource):
            pass

        class TaskFooParams(parameters.TaskParams):
            some_arg = parameters.StrParam('Some arg')

        class TaskBarRes(sdk2.Resource):
            pass

        class TaskBazParams(parameters.TaskParams):
            some_arg = parameters.StrParam('Some arg')

        class FooRunnerTask(RunTaskTemplate):
            class Parameters(get_run_params(TaskFooRes)):
                pass

        class TaskWithFooParams(sdk2.Task):
            class Parameters(get_run_params(TaskFooParams)):
                pass

        class FooRunnerTaskWithParams(RunTaskTemplate):
            class Parameters(get_run_params(TaskFooRes, TaskFooParams)):
                pass

        class FooBarCommonRunnerTask(RunTaskTemplate):
            class Parameters(get_run_params([
                (TaskFooRes, TaskFooParams),
                TaskBarRes,
                FunctionTaskInitializer('runner_name', TaskBazParams, lambda task_params: print(task_params)),
            ])):
                pass

    :type task_initializer_or_params_class_or_resource_or_list: TaskInitializer
        | type[parameters.TaskParams]
        | type[sdk2.Resource]
        | list[
          TaskInitializer
          | tuple[type[sdk2.Resource] | str, type[parameters.TaskParams] | None]
          | type[sdk2.Resource]
        ]
    :type task_params_class: type[parameters.TaskParams]
    :type base_class: type[sdk2.Parameters]
    :rtype: type[sdk2.Parameters]
    """
    if isinstance(task_initializer_or_params_class_or_resource_or_list, list):
        task_type_name_to_task_initializer = collections.OrderedDict([
            TaskInitializer.create_from(args).as_kv_tuple()
            for args in task_initializer_or_params_class_or_resource_or_list
        ])

        class Parameters(base_class):
            _locals = set(inspect.currentframe().f_locals.keys())
            __task_rope_params__ = None
            __task_specific_sb_params__ = defaultdict(list)

            with sdk2.parameters.RadioGroup('Task type to run') as _task_type:
                for task_type_key, task_executor in task_type_name_to_task_initializer.items():
                    setattr(_task_type.values, task_type_key,
                            _task_type.Value(task_executor.task_name))

            __task_type_name_to_task_initializer__ = task_type_name_to_task_initializer

            for _task_type_key, _task_executor in task_type_name_to_task_initializer.items():
                _task_prefix = _task_executor.task_prefix
                with _task_type.value[_task_executor.task_key]:
                    for _param_data in _task_executor.iter_sdk2_executor_param_data():
                        if hasattr(base_class, _param_data.name):
                            continue
                        if _param_data.depend_on:
                            with (
                                parameters.get_sdk2_local_parameter(
                                    _task_prefix + _param_data.depend_on.param_name
                                )
                                or parameters.get_sdk2_local_parameter(
                                _param_data.depend_on.param_name
                            )
                            ).value[_param_data.depend_on.param_value]:
                                sdk2.helpers.set_parameter(
                                    _task_prefix + _param_data.name,
                                    _param_data.param_class(**_param_data.param_kwargs),
                                )
                        else:
                            sdk2.helpers.set_parameter(
                                _task_prefix + _param_data.name,
                                _param_data.param_class(**_param_data.param_kwargs),
                            )
            rm_tmp_vars_from_frame_locals(set(inspect.currentframe().f_locals.keys()) - _locals)

        return Parameters
    elif isinstance(task_initializer_or_params_class_or_resource_or_list, TaskInitializer):
        task_initializer = task_initializer_or_params_class_or_resource_or_list
        task_params_class = task_initializer.task_params_class
        sdk2_param_data_iter = task_initializer.iter_sdk2_executor_param_data()
    elif isinstance(task_initializer_or_params_class_or_resource_or_list, type):
        if issubclass(task_initializer_or_params_class_or_resource_or_list, parameters.TaskParams):
            task_initializer = None
            task_params_class = task_initializer_or_params_class_or_resource_or_list
            sdk2_param_data_iter = task_params_class.iter_sdk2_task_specific_param_data()
        elif issubclass(task_initializer_or_params_class_or_resource_or_list, sdk2.Resource):
            resource_type = task_initializer_or_params_class_or_resource_or_list
            task_initializer = BinaryResourceTaskInitializer(resource_type, task_params_class)
            sdk2_param_data_iter = task_initializer.iter_sdk2_executor_param_data()
        else:
            raise TypeError('Unsupported type {}'.format(task_initializer_or_params_class_or_resource_or_list))
    else:
        raise TypeError('Unsupported type {!r}'.format(task_initializer_or_params_class_or_resource_or_list))

    class Parameters(base_class):
        _locals = set(inspect.currentframe().f_locals.keys())
        __task_rope_params__ = task_params_class

        if task_initializer:
            _task_type = task_initializer.task_key

            __task_type_name_to_task_initializer__ = {_task_type: task_initializer}

        for _param_data in sdk2_param_data_iter:
            if _param_data.depend_on:
                with parameters.get_sdk2_local_parameter(
                    _param_data.depend_on.param_name,
                ).value[_param_data.depend_on.param_value]:
                    sdk2.helpers.set_parameter(
                        _param_data.name,
                        _param_data.param_class(**_param_data.param_kwargs))
            else:
                sdk2.helpers.set_parameter(
                    '_task_executable' if _param_data.name == EXECUTABLE_PARAM_NAME else _param_data.name,
                    _param_data.param_class(**_param_data.param_kwargs))
        rm_tmp_vars_from_frame_locals(set(inspect.currentframe().f_locals.keys()) - _locals)

    return Parameters


class TaskParamsTemplate(sdk2.Task):
    logger = logging.getLogger('task runner')

    @cached_property
    def task_initializer(self):
        """
        :rtype: TaskInitializer | None
        """
        task_type_name_to_task_initializer = getattr(self.Parameters, '__task_type_name_to_task_initializer__', None)
        if task_type_name_to_task_initializer:
            return task_type_name_to_task_initializer[self.Parameters._task_type]  # noqa

    @cached_property
    def task_params_class(self):
        """
        :rtype: type[parameters.TaskParams] | None
        """
        if self.task_initializer:
            return self.task_initializer.task_params_class
        return getattr(self.Parameters, '__task_rope_params__', None)

    @cached_property
    def task_params(self):
        """
        :rtype: parameters.TaskParams | None
        """
        if self.task_initializer:
            return self.task_initializer.get_task_params(self.Parameters, self.Context)
        if self.task_params_class:
            return self.task_params_class.from_sdk2_params(  # noqa
                self.Parameters,
                context=self.Context,
            )

    @cached_property
    def calculated_sandbox_param_defaults(self):
        self.task_params  # noqa
        calculated_default_fields = getattr(self.Context,
                                            parameters.CALCULATED_DEFAULT_FIELDS_NAME, None)
        if not calculated_default_fields:
            return {}
        return {
            field: getattr(self.Context, field) for field in calculated_default_fields
            if hasattr(self.Context, field)
        }

    def on_prepare(self):
        super(TaskParamsTemplate, self).on_prepare()
        # to calc defaults and save them to context
        self.task_params  # noqa


class RunTaskTemplate(sdk2.Task):
    logger = logging.getLogger('task runner')

    def has_task_type(self):
        return bool(self.Parameters._task_type)  # noqa

    @cached_property
    def task_initializer(self):
        """
        :rtype: TaskInitializer
        """
        return self.Parameters.__task_type_name_to_task_initializer__[self.Parameters._task_type]  # noqa

    @cached_property
    def executable_resource_type(self):
        return sdk2.Resource[self.Parameters._task_type]  # noqa

    @cached_property
    def task_params_class(self):
        """
        :rtype: type[parameters.TaskParams] | None
        """
        return self.task_initializer.task_params_class

    @cached_property
    def task_res_name(self):
        return str(self.executable_resource_type)

    @cached_property
    def task_name(self):
        return self.task_initializer.task_name

    @cached_property
    def task_params(self):
        return self.task_initializer.get_task_params(self.Parameters, self.Context)

    @cached_property
    def calculated_sandbox_param_defaults(self):
        self.task_params  # noqa
        calculated_default_fields = getattr(self.Context,
                                            parameters.CALCULATED_DEFAULT_FIELDS_NAME, None)
        if not calculated_default_fields:
            return {}
        return {
            field: getattr(self.Context, field) for field in calculated_default_fields
            if hasattr(self.Context, field)
        }

    def process_task_error(self, ex):
        """
        Process task exception and stop exception propagation if needed by returning True
        :param ex: Exception
        :rtype: bool | None
        """
        self.logger.info('process_task_error %s %s %s', ex, threading.currentThread(), os.getpid())
        sys.stderr.write('{} FAILED'.format(self.task_name.upper()))

    @property
    def current_tags(self):
        tags = []
        for tag in self.Parameters.tags:
            tags.append(str(tag))
        return tags

    def append_tag(self, tag):
        tag_formatted = tag.upper()
        if tag_formatted not in self.current_tags:
            self.Parameters.tags.append(tag_formatted)

    def on_save(self):
        if self.has_task_type():
            self.append_tag(self.task_name)

    def on_prepare(self):
        super(RunTaskTemplate, self).on_prepare()
        # to calc defaults and save them to context
        self.task_params  # noqa

    def _on_stop(self):
        self.logger.info('_on_stop')
        self.task_initializer.on_stop(self, logger=self.logger)

    def on_execute(self):
        executor = self.task_initializer.get_executor(self, logger=self.logger)
        try:
            executor.run()
        except Exception as ex:
            self.process_task_error(ex)
            raise

    def on_terminate(self):
        self.logger.info('on_terminate() %s %s', threading.currentThread(), os.getpid())
        super(RunTaskTemplate, self).on_terminate()
        with self.memoize_stage.on_stop():
            self._on_stop()

    def on_before_timeout(self, seconds):
        logging.info('on_before_timeout(seconds=%s) %s %s', seconds, threading.currentThread(), os.getpid())
        super(RunTaskTemplate, self).on_before_timeout(seconds)
        logging.warning('[TC] %s seconds left until task timeout', seconds)
        if seconds == min(self.timeout_checkpoints()):
            self.set_info('<b>Stop on timeout!</b>', do_escape=False)
            with self.memoize_stage.on_stop():
                self._on_stop()

    def on_timeout(self, prev_status):
        super(RunTaskTemplate, self).on_timeout(prev_status)
        logging.info('on_timeout(prev_status=%s) %s %s', prev_status, threading.currentThread(), os.getpid())
        with self.memoize_stage.on_stop():
            self._on_stop()

    def on_failure(self, prev_status):
        super(RunTaskTemplate, self).on_failure(prev_status)
        logging.info('on_failure(prev_status=%s) %s %s', prev_status, threading.currentThread(), os.getpid())
        with self.memoize_stage.on_stop():
            self._on_stop()

    def on_break(self, prev_status, status):
        super(RunTaskTemplate, self).on_break(prev_status, status)
        logging.info('on_break(prev_status=%s, status=%s) %s %s', prev_status, status,
                     threading.currentThread(), os.getpid())
        with self.memoize_stage.on_stop():
            self._on_stop()
