# -*- coding: utf-8 -*-
import abc
import argparse
import inspect
import itertools
import json
import logging
import os
import types
from collections import namedtuple
from collections import OrderedDict
import datetime

import dateutil.tz
import sandbox.sdk2 as sdk2
import six

from sandbox.projects.rope.compat import ensure_text
from sandbox.projects.rope.compat import text
from sandbox.projects.rope.toposort import toposort
import enum

MISSING = type('Missing', (object,), {'__repr__': lambda self: 'MISSING',
                                      '__bool__': lambda self: False})()

JUST_RETURN = (lambda x: x)

STRICT_DATE_FORMAT = '%Y-%m-%d'

DATE_FORMATS = [
    '%Y',
    '%Y-%m',
    STRICT_DATE_FORMAT,
]

STRICT_DATE_TIME_T_FORMAT = '%Y-%m-%dT%H:%M:%S'

DATE_TIME_T_FORMATS = [
    '%Y-%m-%dT%H',
    '%Y-%m-%dT%H:%M',
    STRICT_DATE_TIME_T_FORMAT,
    '%Y-%m-%dT%H:%M:%S%z',
    '%Y-%m-%dT%H:%M:%S.%f',
    '%Y-%m-%dT%H:%M:%S.%f%z',
]

STRICT_DATE_TIME_S_FORMAT = '%Y-%m-%d %H:%M:%S'

DATE_TIME_S_FORMATS = [
    '%Y-%m-%d %H',
    '%Y-%m-%d %H:%M',
    STRICT_DATE_TIME_S_FORMAT,
    '%Y-%m-%d %H:%M:%S%z',
    '%Y-%m-%d %H:%M:%S.%f',
    '%Y-%m-%d %H:%M:%S.%f%z',
]

DATE_TIME_FORMATS = DATE_FORMATS + DATE_TIME_T_FORMATS + DATE_TIME_S_FORMATS

MOSCOW_TZ = dateutil.tz.gettz('Europe/Moscow')
LOCAL_TZ = dateutil.tz.tzlocal()

CALCULATED_DEFAULT_FIELDS_NAME = '__calculated_defaults__'


class ParamSrc(enum.IntEnum):
    NONE = 0
    D = DIRECTLY_INIT = 1
    C = CLI = 2
    E = ENV = 4
    S = SANDBOX_SDK2_PARAM = 8
    CLI_OR_ENV = C | E
    NOT_SKD2_PARAM = D | C | E
    ALL = D | C | S | E


def is_none_or_empty(value):
    """
    :type value: Any
    :rtype: bool
    """
    return value is None \
           or (isinstance(value, (text, bytes)) and not value.strip()) \
           or (isinstance(value, (list, dict, set, tuple)) and not value)


def get_sdk2_local_parameter(name, default=MISSING):
    """
    Helper function for dynamically getting task parameters defying in curren scope by it's name

    Example:
    .. code-block:: python
        from sandbox import sdk2

        class Parameters(sdk2.Parameters):
            for name in ("a", "b", "c"):
                sdk2.helpers.set_parameter(name, sdk2.parameters.String(name.capitalize()))
            with get_sdk2_local_parameter("b").value['']:
                ...

    :param str name: parameter name
    :type default: Any
    """
    frame = inspect.currentframe().f_back
    if default is MISSING:
        return frame.f_locals[name]
    return frame.f_locals.get(name, default=default)


def camel_to_kebab_case(name):
    return '-'.join(name.split('_'))


def camel_case_to_cli_arg(name):
    name = camel_to_kebab_case(name)
    if len(name) > 1:
        return '--' + name
    return '-' + name


def kebab_to_camel_case(name):
    return '_'.join(name.split('-'))


class SecretValue(str):
    # noinspection PyInitNewSignature
    def __new__(cls, str_content, description=''):
        # noinspection PyArgumentList
        self = str.__new__(cls, str_content)
        self.description = description
        return self

    def __repr__(self):
        return "'<secret: {self.description}>'".format(self=self)


def abs_required(required):
    """
    :type required: bool | ParamSrc | None
    :rtype: ParamSrc
    """
    if isinstance(required, bool):
        return {True: ParamSrc.ALL, False: ParamSrc.NONE}[required]
    else:
        return required or ParamSrc.NONE


class TaskParamAbstract(six.with_metaclass(abc.ABCMeta, object)):

    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: Any | (TaskParams) -> Any
        :type required: bool | ParamSrc
        :type skip_in: ParamSrc
        :type depend_on: tuple[str, Any]
        """
        self.descr = descr
        self.details = details
        self.attr_name = None
        self._cli_arg = cli_arg
        self._env_arg = env_arg
        if default is MISSING:
            self.default = None
            self.default_getter = None
            self.has_default = False
        else:
            self.has_default = True
            if callable(default):
                self.default = None
                self.default_getter = DefaultGetter.ensure_wraps(default)
            else:
                self.default = default
                self.default_getter = None
        self.skip_in = skip_in
        self.required = abs_required(required)
        self.depend_on = DependedOn(*depend_on) if depend_on else depend_on

        # To save param declare order
        frame = inspect.currentframe()
        while frame.f_back and frame.f_code.co_name == '__init__':
            frame = frame.f_back

        props = frame.f_locals.setdefault('__props_order__', [])
        props.append(self)

    def is_definitely_required(self, params=None, src=None):
        """
        :type params: TaskParams | type[sdk2.Task.Parameters]
        :type src: ParamSrc
        :rtype: Any
        """
        if self.has_default:
            return False
        if self.depend_on:
            if (
                not params
                or getattr(params, self.depend_on.param_name, MISSING) != self.depend_on.param_value
            ):
                return False
        if src:
            return self.required & src != 0
        return self.required == ParamSrc.ALL

    def set_attr_name(self, attr_name):
        self.attr_name = attr_name

    def load(self, value, params=None, src=None):
        """
        :type value: str | None
        :type params: TaskParams
        :type src: ParamSrc
        :rtype: Any
        """
        if is_none_or_empty(value):
            if self.is_definitely_required(params=params, src=src):
                raise ValueError('Empty value is not allowed for param {}'.format(self.attr_name))
            else:
                return None
        return self.load_value(value)

    @abc.abstractmethod
    def load_value(self, value):
        """
        :type value: str
        :rtype: Any
        """
        return

    def dump(self, value, params=None):
        """
        :type value: Any
        :type params: TaskParams
        :rtype: str
        """
        if value is None:
            if self.is_definitely_required(params=params):
                raise ValueError('Empty value is not allowed for param {}'.format(self.attr_name))
            else:
                return ''
        return self.dump_value(value)

    @abc.abstractmethod
    def dump_value(self, value):
        """
        :type value: Any
        :rtype: str
        """
        return

    def load_from_params(self, params, prefix=None, context=None):
        """
        deprecated

        :type params: type[sdk2.Task.Parameters]
        :type prefix: str
        :type context: type[sdk2.Task.Context]
        :return: Any
        """
        return self.load_from_sb_sdk2_params(params, prefix=prefix, sb_sdk2_context=context)

    @abc.abstractmethod
    def load_from_sb_sdk2_params(self, sb_sdk2_params, prefix=None, sb_sdk2_context=None):
        """
        :type sb_sdk2_params: type[sdk2.Task.Parameters]
        :type prefix: str
        :type sb_sdk2_context: type[sdk2.Task.Context]
        :return: Any
        """
        return

    @abc.abstractmethod
    def iter_sdk2_param_data(self, name):
        """
        :type name: str
        :rtype: typing.Generator[Sdk2ParamCreationData]
        """
        yield

    @abc.abstractmethod
    def proxy_params(self, value, params, prefix=None):
        """
        :type value: Any
        :type params: type[sdk2.Task.Parameters]
        :type prefix: str
        :rtype: dict[str, Any]
        """
        return

    @property
    def cli_arg(self):
        assert self._cli_arg or self.attr_name
        return self._cli_arg or camel_case_to_cli_arg(self.attr_name)

    @property
    def env_arg(self):
        assert self._env_arg or self.attr_name
        return self._env_arg or self.attr_name.upper()

    @property
    def cli_arg_var_name(self):
        return kebab_to_camel_case(self.cli_arg.lstrip('-'))

    @property
    def argparse_margs(self):
        kwargs = dict(dest=self.attr_name, help=self.descr)
        if self.has_default:
            kwargs.update(default=self.default)
        return (self.cli_arg,), kwargs

    def __repr__(self):
        return (
            '{self.__class__.__name__}('
            'attr_name={self.attr_name!r}'
            ', descr={self.descr!r})'.format(self=self)
        )


class SimpleTaskParamAbstract(six.with_metaclass(abc.ABCMeta, TaskParamAbstract)):
    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        sdk2_param_class=None,
        sdk2_param_kwargs=None,
        sdk2_param_proxy=None,
        load_from_sdk2_params=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: Any | (TaskParams) -> Any
        :type required: bool | ParamSrc
        :type sdk2_param_class: type[sdk2.parameters.internal.parameters.Parameter]
            | dict[str, type[sdk2.parameters.internal.parameters.Parameter]]
        :type sdk2_param_kwargs: dict[str] | dict[str, dict[str]]
        :param sdk2_param_proxy: function to proxy param value back to source sandbox parameters
        :type sdk2_param_proxy: (Any, dict[str, (str, sdk2.parameters.internal.parameters.Parameter)]
            | (str, sdk2.parameters.internal.parameters.Parameter)) -> dict[str, Any]
        :type load_from_sdk2_params: (Any | dict[str, Any]) -> Any
        :type skip_in: ParamSrc
        :type depend_on: tuple[str, Any]
        """
        super(SimpleTaskParamAbstract, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            default=default,
            required=required,
            skip_in=skip_in,
            depend_on=depend_on,
        )
        self._sdk2_param_class = sdk2_param_class
        self._sdk2_param_kwargs = sdk2_param_kwargs

        sdk2_config = [sdk2_param_class]
        if sdk2_param_kwargs:
            sdk2_config.append(sdk2_param_kwargs)

        def proxy_value_with_original_sdk2_name(value, name_n_param):
            return {name_n_param[0]: self.dump(value)}

        self._multi_param_names = None
        self._sdk2_param_proxy = sdk2_param_proxy or proxy_value_with_original_sdk2_name
        if isinstance(sdk2_param_class, dict):
            multi_param_names = list(sdk2_param_class.keys())
            if sdk2_param_kwargs:
                assert set(sdk2_param_kwargs.keys()).issubset(set(multi_param_names)), (
                    'sdk2_param_kwargs should be a dict of kwargs for sdk2_param_class dict'
                )
            self._multi_param_names = multi_param_names
            assert load_from_sdk2_params is not None, (
                'load_from_sdk2_params should be specified for multi param value'
            )

        self._load_from_sdk2_params = load_from_sdk2_params or JUST_RETURN

    def _get_sb_sdk2_params_with_names(self, sb_sdk2_params, prefix=None):
        """
        Return single pair of final sandbox sdk2 param name and its value,
        or several pairs if param deriving from several sandbox sdk2 param
        in the form of sub_param_name -> pair dict

        :type sb_sdk2_params: type[sdk2.Task.Parameters]
        :type prefix: str
        :rtype: dict[str, (str, Any)] | (str, Any)
        """
        name = self.attr_name
        prefix_name = '{}{}'.format(prefix, self.attr_name) if prefix else None

        def _get_param(suffix=None):
            # name with suffix
            s_name = '{}_{}'.format(name, suffix) if suffix else name
            if not prefix_name:
                return s_name, getattr(sb_sdk2_params, s_name, None)
            # name with prefix and suffix
            p_s_name = '{}_{}'.format(prefix_name, suffix) if suffix else prefix_name
            if hasattr(sb_sdk2_params, s_name):
                if hasattr(sb_sdk2_params, p_s_name):
                    raise ValueError('Param name conflict: both {} and {} exists'.format(
                        s_name, p_s_name))
                return s_name, getattr(sb_sdk2_params, s_name)
            return p_s_name, getattr(sb_sdk2_params, p_s_name, None)

        if self._multi_param_names:
            return {
                sfx: _get_param(sfx) for sfx in self._multi_param_names
            }
        return _get_param()

    def _get_param_value(self, sb_sdk2_params, prefix=None):
        """
        Return param value or sub-param values in a form of sub_param_name -> value
        :type sb_sdk2_params: type[sdk2.Task.Parameters]
        :type prefix: str
        :rtype: dict[str, Any] | Any
        """
        p = self._get_sb_sdk2_params_with_names(sb_sdk2_params, prefix=prefix)
        if isinstance(p, dict):
            return {k: v[1] for k, v in p.items()}
        return p[1]

    def _get_from_sb_sdk2_context(self, sb_sdk2_context, prefix=None):
        calced_default = getattr(sb_sdk2_context, CALCULATED_DEFAULT_FIELDS_NAME, None) or []
        if not calced_default:
            return MISSING
        if prefix and (prefix + self.attr_name) in calced_default:
            return getattr(sb_sdk2_context, prefix + self.attr_name)
        if self.attr_name in calced_default:
            return getattr(sb_sdk2_context, self.attr_name)
        return MISSING

    def _get_sb_sdk2_param(self, sb_sdk2_params, name, prefix=None):
        prefix_name = '{}{}'.format(prefix, name) if prefix else None
        if prefix_name and hasattr(sb_sdk2_params, prefix_name):
            return getattr(sb_sdk2_params, prefix_name, None)
        return getattr(sb_sdk2_params, name, None)

    def load_from_sb_sdk2_params(self, sb_sdk2_params, prefix=None, sb_sdk2_context=None):
        """
        :type sb_sdk2_params: type[sdk2.Task.Parameters]
        :type prefix: str
        :type sb_sdk2_context: type[sdk2.Task.Context]
        :return: Any
        """
        if (
            self.depend_on
            and self._get_sb_sdk2_param(
                sb_sdk2_params, self.depend_on.param_name, prefix=prefix
            ) != self.depend_on.param_value
        ):
            return None if self.default is None else self.default
        default = self.default
        if self.skip_in & ParamSrc.SANDBOX_SDK2_PARAM:
            return default
        if sb_sdk2_context:
            value_from_context = self._get_from_sb_sdk2_context(sb_sdk2_context, prefix=prefix)
            if value_from_context is not MISSING:
                return value_from_context
        value = self._load_from_sdk2_params(self._get_param_value(sb_sdk2_params, prefix=prefix))
        return default if is_none_or_empty(value) else value

    def _iter_sdk2_param_class(self):
        if isinstance(self._sdk2_param_class, dict):
            for suffix, sdk2_param_class in self._sdk2_param_class.items():
                yield suffix, sdk2_param_class, self._sdk2_param_kwargs.get(suffix, None)
        else:
            yield None, self._sdk2_param_class or sdk2.parameters.String, self._sdk2_param_kwargs

    def iter_sdk2_param_data(self, name):
        """
        :type name: str
        :rtype: typing.Generator[Sdk2ParamCreationData]
        """
        for n, (sfx, sdk2_prm_class, sdk2_prm_kwargs) in enumerate(self._iter_sdk2_param_class()):
            param_kwargs = dict(
                label='{} {}'.format(self.descr, sfx.replace('_', ' ')) if sfx else self.descr,
            )
            if self.details and (sfx is None or n == 0):
                param_kwargs.update(description=self.details)
            if self._load_from_sdk2_params is JUST_RETURN:
                if self.has_default and self.default is not None:
                    param_kwargs.update(default=self.default)
                if self.required & ParamSrc.SANDBOX_SDK2_PARAM:
                    param_kwargs.update(required=True)
            if sdk2_prm_kwargs:
                param_kwargs.update(sdk2_prm_kwargs)

            param_name = '{}_{}'.format(name, sfx) if sfx else name
            param_data = Sdk2ParamCreationData(param_name, param_class=sdk2_prm_class,
                                               param_kwargs=param_kwargs, depend_on=self.depend_on)

            yield param_data

    def proxy_params(self, value, params, prefix=None):
        """
        :type value: Any
        :type params: type[sdk2.Task.Parameters]
        :type prefix: str
        :rtype: dict[str, Any]
        """
        if self._sdk2_param_proxy:
            return self._sdk2_param_proxy(value, self._get_sb_sdk2_params_with_names(params, prefix=prefix))
        key = prefix + self.attr_name if prefix else self.attr_name
        return {key: self.dump(value)}


class BoolParam(SimpleTaskParamAbstract):

    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        sdk2_param_kwargs=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: bool | (TaskParams) -> bool
        :type required: bool | ParamSrc
        :type sdk2_param_kwargs: dict[str]
        :type skip_in: ParamSrc
        :type depend_on: tuple[str, Any]
        """
        super(BoolParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            default=default,
            required=required,
            sdk2_param_class=sdk2.parameters.Bool,
            sdk2_param_kwargs=sdk2_param_kwargs,
            skip_in=skip_in,
            depend_on=depend_on,
        )

    def load_value(self, value):
        """
        :type value: bool | str
        :rtype: bool | None
        """
        if isinstance(value, (text, bytes)):
            if value.isdigit():
                return bool(int(value))
            v = value.strip().lower()
            if v in {'false', 'f'}:
                return False
            elif v in {'true', 't'}:
                return True
            else:
                raise ValueError('Do not know how to interpret {!r} as bool'.format(value))
        elif isinstance(value, bool):
            return value

    def dump_value(self, value):
        """
        :type value: bool
        :rtype: str | None
        """
        return str(int(value))


class StrParam(SimpleTaskParamAbstract):

    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        sdk2_param_class=None,
        sdk2_param_kwargs=None,
        sdk2_param_proxy=None,
        load_from_sdk2_params=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: str | (TaskParams) -> str
        :type required: bool | ParamSrc
        :type sdk2_param_class: type[sdk2.parameters.internal.parameters.Parameter]
            | dict[str, type[sdk2.parameters.internal.parameters.Parameter]]
        :type sdk2_param_kwargs: dict[str] | dict[str, dict[str]]
        :type sdk2_param_proxy: (Any, dict[str, (str, sdk2.parameters.internal.parameters.Parameter)]
            | (str, sdk2.parameters.internal.parameters.Parameter)) -> dict[str, Any]
        :type load_from_sdk2_params: (Any | dict[str, Any]) -> str | None
        :type skip_in: ParamSrc
        :type depend_on: tuple[str, Any]
        """
        super(StrParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            default=default,
            required=required,
            sdk2_param_class=sdk2_param_class,
            sdk2_param_kwargs=sdk2_param_kwargs,
            sdk2_param_proxy=sdk2_param_proxy,
            load_from_sdk2_params=load_from_sdk2_params,
            skip_in=skip_in,
            depend_on=depend_on,
        )

    def load_value(self, value):
        """
        :type value: str
        :rtype: str
        """
        return str(value).strip()

    def dump_value(self, value):
        """
        :type value: str
        :rtype: str
        """
        return str(value)


class SecretParam(StrParam):
    def load_value(self, value):
        """
        :type value: str
        :rtype: SecretValue
        """
        return SecretValue(value, description=self.attr_name)


class IntParam(SimpleTaskParamAbstract):

    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        sdk2_param_class=None,
        sdk2_param_kwargs=None,
        sdk2_param_proxy=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: int | (TaskParams) -> int
        :type required: bool | ParamSrc
        :type sdk2_param_class: type[sdk2.parameters.internal.parameters.Parameter]
        :type sdk2_param_kwargs: dict[str]
        :type sdk2_param_proxy: (Any, dict[str, (str, sdk2.parameters.internal.parameters.Parameter)]
            | (str, sdk2.parameters.internal.parameters.Parameter)) -> dict[str, Any]
        :type skip_in: ParamSrc
        :type depend_on: tuple[str, Any]
        """
        super(IntParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            default=default,
            required=required,
            sdk2_param_class=sdk2_param_class,
            sdk2_param_kwargs=sdk2_param_kwargs,
            sdk2_param_proxy=sdk2_param_proxy,
            skip_in=skip_in,
            depend_on=depend_on,
        )

    def load_value(self, value):
        """
        :type value: str | int
        :rtype: int
        """
        if isinstance(value, int):
            return value
        return int(value)

    def dump_value(self, value):
        """
        :type value: int
        :rtype: str
        """
        return str(value)


class DateParam(SimpleTaskParamAbstract):
    _date_time_class = datetime.date

    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        sdk2_param_class=None,
        sdk2_param_kwargs=None,
        sdk2_param_proxy=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
        fmt=None,
        tzinfo=MOSCOW_TZ,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: datetime.date | (TaskParams) -> datetime.date
        :type required: bool | ParamSrc
        :type sdk2_param_class: type[sdk2.parameters.internal.parameters.Parameter]
        :type sdk2_param_kwargs: dict[str]
        :type sdk2_param_proxy: (Any, dict[str, (str, sdk2.parameters.internal.parameters.Parameter)]
            | (str, sdk2.parameters.internal.parameters.Parameter)) -> dict[str, Any]
        :type skip_in: ParamSrc
        :type depend_on: tuple[str, Any]
        :type fmt: str | list(str)
        :type tzinfo: datetime.tzinfo | None
        """
        super(DateParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            default=default,
            required=required,
            sdk2_param_class=sdk2_param_class,
            sdk2_param_kwargs=sdk2_param_kwargs,
            sdk2_param_proxy=sdk2_param_proxy,
            skip_in=skip_in,
            depend_on=depend_on,
        )
        self.tzinfo = tzinfo
        fmt = fmt or DATE_FORMATS
        self.formats = fmt if isinstance(fmt, list) or isinstance(fmt, tuple) else [fmt]

    def _fromtimestamp(self, value):
        result_datetime = datetime.datetime.fromtimestamp(value)
        if self.tzinfo:
            result_datetime = result_datetime.replace(tzinfo=LOCAL_TZ).astimezone(self.tzinfo)
        return result_datetime.date()

    def _strptime(self, value, fmt):
        return datetime.datetime.strptime(ensure_text(value), fmt).date()

    def load_value(self, value):
        """
        :tyupe value: int | str
        :rtype: datetime.date
        """
        if isinstance(value, int) or (isinstance(value, (text, bytes)) and value.isdigit()):
            return self._fromtimestamp(int(value))
        elif isinstance(value, (text, bytes)):
            for fmt in self.formats:
                try:
                    return self._strptime(value, fmt)
                except ValueError:
                    pass
            raise ValueError('Unsupported {} format {!r} for param {}'.format(
                self._date_time_class.__name__, value, self.attr_name))
        elif isinstance(value, self._date_time_class):
            return value
        else:
            raise ValueError('Unsupported {} type {!r} for param {}'.format(
                self._date_time_class.__name__, value, self.attr_name))

    def dump_value(self, value):
        """
        :type value: datetime.date
        :rtype: str
        """
        return value.strftime(STRICT_DATE_FORMAT)


class DateTimeParam(DateParam):
    _date_time_class = datetime.datetime

    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        sdk2_param_class=None,
        sdk2_param_kwargs=None,
        sdk2_param_proxy=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
        fmt=None,
        tzinfo=MOSCOW_TZ,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: datetime.datetime | (TaskParams) -> datetime.datetime
        :type required: bool | ParamSrc
        :type sdk2_param_class: type[sdk2.parameters.internal.parameters.Parameter]
        :type sdk2_param_kwargs: dict[str]
        :type sdk2_param_proxy: (Any, dict[str, (str, sdk2.parameters.internal.parameters.Parameter)]
            | (str, sdk2.parameters.internal.parameters.Parameter)) -> dict[str, Any]
        :type skip_in: ParamSrc
        :type depend_on: tuple[str, Any]
        :type fmt = str | list(str)
        :type tzinfo: datetime.tzinfo | None
        """
        fmt = fmt or DATE_TIME_FORMATS
        super(DateTimeParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            default=default,
            required=required,
            sdk2_param_class=sdk2_param_class,
            sdk2_param_kwargs=sdk2_param_kwargs,
            sdk2_param_proxy=sdk2_param_proxy,
            skip_in=skip_in,
            depend_on=depend_on,
            fmt=fmt,
            tzinfo=tzinfo,
        )

    def _fromtimestamp(self, value):
        result = self._date_time_class.fromtimestamp(value)
        if self.tzinfo:
            return result.replace(tzinfo=LOCAL_TZ).astimezone(self.tzinfo)
        return result

    def _strptime(self, value, fmt):
        result = self._date_time_class.strptime(ensure_text(value), fmt)
        if self.tzinfo:
            return result.replace(tzinfo=self.tzinfo)
        return result

    def dump_value(self, value):
        """
        :type value: datetime.datetime
        :rtype: str
        """
        result = value.strftime('%Y-%m-%d %H:%M:%S.%f')
        for end in [
            ' 00:00:00.000000',
            ':00.000000',
            '.000000',
        ]:
            if result.endswith(end):
                result = result[:-len(end)]
                break
        return result


def proxy_single_src_param(_, name_n_param):
    return {name_n_param[0]: str(name_n_param[1])}


def proxy_multiple_src_params(_, sfx_to_name_n_param):
    return {name: param for _, (name, param) in sfx_to_name_n_param.items()}


class VaultSecretParam(SecretParam):

    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default_vault_name=None,
        default_vault_user=None,
        specify_owner=False,
        required=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :type required: bool | ParamSrc
        :type skip_in: ParamSrc
        :type default_vault_name: str
        :type default_vault_user: str
        :type specify_owner: bool
        :type depend_on: tuple[str, Any]
        """
        if not details and not specify_owner:
            details = 'for vault owner {}'.format(default_vault_user)
        required = abs_required(required)
        sdk2_param_class = OrderedDict(dict(
            vault_name=sdk2.parameters.String,
        ).items())
        sdk2_param_kwargs = OrderedDict(dict(
            vault_name=dict(
                required=bool(required & ParamSrc.SANDBOX_SDK2_PARAM),
                default=default_vault_name,
            ),
        ).items())

        sdk2_param_proxy = proxy_single_src_param

        if specify_owner:
            sdk2_param_class.update(dict(
                vault_user=sdk2.parameters.String,
            ))
            sdk2_param_kwargs.update(dict(
                vault_user=dict(
                    default=default_vault_user,
                ),
            ))

            sdk2_param_proxy = proxy_multiple_src_params

        def load_from_sdk2_params(sdk2_params):
            """
            :type sdk2_params: dict[str, str]
            :rtype: str | None
            """
            vault_name = sdk2_params['vault_name']
            if not vault_name:
                return None

            vault_user = sdk2_params.get('vault_user', None) or self.default_vault_user
            if vault_user:
                return sdk2.Vault.data(vault_user, vault_name)

            return sdk2.Vault.data(vault_name)

        super(VaultSecretParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            required=required,
            skip_in=skip_in,
            depend_on=depend_on,
            sdk2_param_class=sdk2_param_class,
            sdk2_param_kwargs=sdk2_param_kwargs,
            sdk2_param_proxy=sdk2_param_proxy,
            load_from_sdk2_params=load_from_sdk2_params,
        )
        self.default_vault_name = default_vault_name
        self.default_vault_user = default_vault_user
        self.specify_owner = specify_owner


class YavSecretParam(SecretParam):

    def __init__(
        self,
        descr,
        default_yav_secret_key,
        details=None,
        cli_arg=None,
        env_arg=None,
        default_yav_secret=None,
        default_yav_secret_version=None,
        required=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type default_yav_secret_key: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :type required: bool | ParamSrc
        :type skip_in: ParamSrc
        :type default_yav_secret: str
        :type depend_on: tuple[str, Any]
        """
        sdk2_param_kwargs = dict()
        if default_yav_secret:
            if default_yav_secret_version:
                default_yav_secret = '@'.join((default_yav_secret, default_yav_secret_version))
            if default_yav_secret_key:
                default_yav_secret = '#'.join((default_yav_secret, default_yav_secret_key))
            sdk2_param_kwargs.update(default=default_yav_secret)
        elif descr is None:
            descr = '^ "{}" key by default'.format(default_yav_secret_key)

        def load_from_sdk2_params(sdk2_param):
            """
            :type sdk2_param: sdk2.yav.Secret | None
            :rtype: str | None
            """
            if not sdk2_param:
                return None
            return sdk2_param.data()[sdk2_param.default_key or default_yav_secret_key]

        super(YavSecretParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            required=required,
            skip_in=skip_in,
            depend_on=depend_on,
            sdk2_param_class=sdk2.parameters.YavSecret,
            sdk2_param_kwargs=sdk2_param_kwargs,
            sdk2_param_proxy=lambda _, p: {p[0]: str(p[1].secret)},
            load_from_sdk2_params=load_from_sdk2_params,  # noqa
        )


class ListParam(SimpleTaskParamAbstract):
    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        sdk2_param_kwargs=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_ar: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: list | (TaskParams) -> list
        :type required: bool | ParamSrc
        :type skip_in: ParamSrc
        :type sdk2_param_kwargs: dict[str]
        :type depend_on: tuple[str, Any]
        """
        super(ListParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            default=default,
            required=required,
            sdk2_param_class=sdk2.parameters.List,
            sdk2_param_kwargs=sdk2_param_kwargs,
            skip_in=skip_in,
            depend_on=depend_on,
        )

    def load_value(self, value):
        """
        :tyupe value: list | str | None
        :rtype: list | None
        """
        if not (value.strip() if isinstance(value, str) else value):
            return None

        parsed_value = list(value)
        return parsed_value

    def dump_value(self, value):
        """
        :type value: list
        :rtype: str
        """
        return ' '.join(value)


class DictParam(SimpleTaskParamAbstract):

    def __init__(
        self,
        descr,
        details=None,
        cli_arg=None,
        env_arg=None,
        default=MISSING,
        required=None,
        sdk2_param_kwargs=None,
        skip_in=ParamSrc.NONE,
        depend_on=None,
    ):
        """
        :type descr: str
        :type details: str
        :param cli_arg: command line argument name (eg: '--some-name')
        :type cli_arg: str
        :param env_arg: name of environment variable (eg: 'SOME_NAME')
        :type env_arg: str
        :param default: default
        :type default: dict | (TaskParams) -> dict
        :type required: bool | ParamSrc
        :type skip_in: ParamSrc
        :type sdk2_param_kwargs: dict[str]
        :type depend_on: tuple[str, Any]
        """
        super(DictParam, self).__init__(
            descr=descr,
            details=details,
            cli_arg=cli_arg,
            env_arg=env_arg,
            default=default,
            required=required,
            sdk2_param_class=sdk2.parameters.Dict,
            sdk2_param_kwargs=sdk2_param_kwargs,
            sdk2_param_proxy=lambda v, p: {p[0]: v},
            skip_in=skip_in,
            depend_on=depend_on,
        )

    def load_value(self, value):
        """
        :tyupe value: dict | str | None
        :rtype: dict | None
        """
        if not (value.strip() if isinstance(value, str) else value):
            return None
        elif isinstance(value, (text, bytes)):
            parsed_value = json.loads(value)
            if not isinstance(parsed_value, dict):
                raise ValueError('Is not a valid json dict: {!r} for param {}'.format(
                    parsed_value, self.attr_name))
            return parsed_value
        elif isinstance(value, dict):
            return value
        else:
            raise ValueError('Unsupported date type {!r} for param {}'.format(value, self.attr_name))

    def dump_value(self, value):
        """
        :type value: dict
        :rtype: str
        """
        return json.dumps(value)


class TaskParamsMeta(type):
    def __new__(mcs, name, bases, attrs):
        props_order = attrs.pop('__props_order__', [])
        argparse_margs = attrs['__argparse_margs__'] = attrs.get('__argparse_margs__') or {}
        env_args = attrs['__env_args__'] = attrs.get('__argparse_margs__') or {}
        task_params = attrs['__task_params__'] = attrs.get('__task_params__') or []
        task_params_map = attrs['__task_params_map__'] = attrs.get('__task_params_map__') or {}
        required_args = []
        optional_args = []
        defaults = []

        def iter_task_params():
            task_params_ = [(n, v) for n, v in attrs.items() if isinstance(v, TaskParamAbstract)]

            def get_prop_order(name_n_prop):
                _, prop = name_n_prop
                try:
                    return props_order.index(prop)
                except ValueError:
                    return len(props_order)

            if len(props_order) == len(task_params_):
                task_params_.sort(key=get_prop_order)

            for attr_name, value in task_params_:
                if isinstance(value, TaskParamAbstract):
                    value.set_attr_name(attr_name)
                    yield value

        def iter_base_task_params():
            for base in bases:
                for task_param_ in getattr(base, '__task_params__', None) or []:
                    yield task_param_

        for task_param in itertools.chain(iter_base_task_params(), iter_task_params()):
            task_params.append(task_param)
            if task_param.attr_name in task_params_map:
                raise ValueError('Duplicated param name {}'.format(task_param.attr_name))
            task_params_map[task_param.attr_name] = task_param
            argparse_margs[task_param.attr_name] = task_param.argparse_margs
            env_args[task_param.env_arg] = task_param
            if task_param.is_definitely_required():
                required_args.append(task_param.attr_name)
            else:
                optional_args.append(task_param.attr_name)
                defaults.append(task_param.default)

        # init method to past all values by their attr_names
        args = required_args + optional_args
        if args:
            init_code = compile(
                'def __init__(self, {}, __lazy_defaults_callback__, __src__): {}'
                '; self._calc_lazy_defaults(__lazy_defaults_callback__, __src__)'.format(
                    ','.join(args),
                    ';'.join((
                        'self.{0} = self.__task_params_map__["{0}"]'
                        '.load({0}, params=self, src=__src__)'.format(a)
                        for a in args
                    ))
                ), '<string>', 'exec')
            defaults.append(None)  # as __lazy_defaults_callback__
            defaults.append(ParamSrc.DIRECTLY_INIT)  # as __src__
            attrs['__init__'] = types.FunctionType(
                init_code.co_consts[0], globals(), '__init__', tuple(defaults))

        return type.__new__(mcs, name, bases, attrs)


class DefaultGetterError(Exception):
    pass


class DefaultGetterRequirementsError(DefaultGetterError):
    def __init__(self, unsatisfied_requirements, task_param):
        super(DefaultGetterRequirementsError, self).__init__(
            'Requirements {} for default value of {} param is not satisfied'.format(
                unsatisfied_requirements, task_param.attr_name
            ))
        self.unsatisfied_requirements = unsatisfied_requirements


class DefaultGetter(object):
    def __init__(self, func, require=None):
        """
        :type func: (TaskParams) -> Any
        :type require: str | list[str]
        """
        self.func = func
        self.require = [require] if isinstance(require, (text, bytes)) else require

    @classmethod
    def ensure_wraps(cls, getter):
        if isinstance(getter, cls):
            return getter
        return cls(getter)

    def __call__(self, params, task_param):
        """
        :type params: TaskParams
        :return: TaskPram value
        :rtype: Any
        """

        if self.require:
            req = {r: hasattr(params, r) and getattr(params, r) for r in self.require}
            if not all(req.values()):
                raise DefaultGetterRequirementsError([r for r, v in req.items() if v], task_param)
        return self.func(params)


class TaskParamsTypeHints(object):
    """
    :type description: str
    :type __argparse_margs__: dict[str, tuple[tuple, dict[str]]]
    :param __argparse_margs__: attr name to args, kwargs for proper argparse add_argument call
    :type __env_args__: dict[str, TaskParamAbstract]
    :param __env_args__: env argument name to param
    :type __task_params__: list[TaskParamAbstract]
    :type __task_params_map__: dict[str, TaskParamAbstract]
    """
    description = None
    __argparse_margs__ = None
    __env_args__ = None
    __task_params__ = None
    __task_params_map__ = None

    def __init__(self, *args, **kwargs):
        pass


DependedOn = namedtuple('DependedOn', ['param_name', 'param_value'])


class Sdk2ParamCreationData(object):
    def __init__(self, name, param_class, param_kwargs, depend_on=None):
        """
        :type name: str
        :type param_class: type[sdk2.parameters.internal.parameters.Parameter]
        :type param_kwargs: dict[str]
        :type depend_on: tuple[str, Any] | DependedOn
        """
        self.name = name
        self.param_class = param_class
        self.param_kwargs = param_kwargs
        self.depend_on = DependedOn(*depend_on) if depend_on else depend_on

    def __eq__(self, other):
        if not isinstance(other, Sdk2ParamCreationData):
            return False
        return (
            self.name == other.name and
            self.param_class == other.param_class and
            self.param_kwargs == other.param_kwargs and
            self.depend_on == other.depend_on
        )

    def __repr__(self):
        return (
            '{self.__class__.__name__}(name={self.name}'
            ', param_class={self.param_class}'
            ', param_kwargs={self.param_kwargs}'
            ', depend_on={self.depend_on})'
        ).format(self=self)


class TaskParams(six.with_metaclass(TaskParamsMeta, TaskParamsTypeHints)):

    @classmethod
    def iter_sdk2_task_specific_param_data(cls, prefix=''):
        """
        :type prefix: str
        :rtype: typing.Generator[Sdk2ParamCreationData]
        """
        # TODO: toposort depends_on
        for task_param in cls.__task_params__:
            if task_param.skip_in & ParamSrc.SANDBOX_SDK2_PARAM:
                continue
            name = prefix + task_param.attr_name
            for param_data in task_param.iter_sdk2_param_data(name):
                yield param_data

    @classmethod
    def iter_params_to_calc_defaults(cls, obj, src=None):
        """
        iterate over empty params that need to be calculated by default
        in proper order (as one param calculation could depends on another)
        :type obj: TaskParams
        :type src: ParamSrc
        :rtype: typing.Generator[TaskParamAbstract]
        """
        params_to_calc_defaults = {
            p.attr_name: p for p in cls.__task_params__
            if p.default_getter and getattr(obj, p.attr_name) is None
        }

        def get_requirements_filter(derived_attr_name):
            def filter_requirements(require):
                result = set()
                for attr_name_ in require or []:
                    if attr_name_ in params_to_calc_defaults:
                        result.add(attr_name_)
                    else:
                        value = getattr(obj, attr_name_)
                        if (
                            value is None
                            and cls.__task_params_map__[attr_name_].is_definitely_required(obj, src)
                        ):
                            raise ValueError(
                                'Could not derive default value of {} param as required param {} '
                                'is None'.format(derived_attr_name, attr_name_))
                return result

            return filter_requirements

        for attr_name in itertools.chain(*toposort({
            p.attr_name: get_requirements_filter(p.attr_name)(
                getattr(p.default_getter, 'require', None)
            )
            for p in params_to_calc_defaults.values()
        })):
            yield params_to_calc_defaults[attr_name]

    def _calc_lazy_defaults(self, lazy_defaults_callback=None, src=None):
        """
        :type lazy_defaults_callback: (TaskParamAbstract, Any) -> None
        :type src: ParamSrc
        """
        for task_param in self.iter_params_to_calc_defaults(self, src):
            value = task_param.default_getter(self, task_param)
            setattr(self, task_param.attr_name, value)
            if lazy_defaults_callback:
                lazy_defaults_callback(task_param, value)

    @classmethod
    def from_cli_args_or_env(cls, args=None, env=None, defaults=None):
        """
        To get args only from env use:
        >>> TaskParams.from_cli_args_or_env(args=[])

        To get args only from cli args use:
        >>> TaskParams.from_cli_args_or_env(env={})

        :type args: list[str]
        :type env: dict[str, str]
        :type defaults: dict[str, str]
        :rtype: TaskParams
        """
        env = env or os.environ
        defaults = defaults or {}
        parser = argparse.ArgumentParser(
            cls.description,
            formatter_class=argparse.ArgumentDefaultsHelpFormatter,
        )
        lazy_kwargs = {}
        for attr_name, (args_, kwargs_) in cls.__argparse_margs__.items():
            task_param = cls.__task_params_map__[attr_name]
            env_val_or_default = env.get(task_param.env_arg, defaults.get(task_param.attr_name, None))

            if env_val_or_default:
                if 'default' in kwargs_ and 'required' in kwargs_:
                    del kwargs_['required']
                kwargs_['default'] = env_val_or_default
            parser.add_argument(*args_, **kwargs_)
            lazy_kwargs[attr_name] = (
                lambda task_param_: lambda args__: getattr(args__, task_param_.attr_name)
            )(task_param)
        parsed_args = parser.parse_args() if args is None else parser.parse_args(args)
        kwargs = {k: v(parsed_args) for k, v in lazy_kwargs.items()}
        obj = cls(**dict(kwargs, __src__=ParamSrc.CLI))

        return obj

    @classmethod
    def from_sdk2_params(cls, params, prefix='', common=None, context=None):
        """
        :type params: type[sdk2.Task.Parameters]
        :type prefix: str
        :type common: dict[str]
        :type context: type[sdk2.Task.Context]
        :rtype: TaskParams
        """
        kwargs = {
            p.attr_name: p.load_from_sb_sdk2_params(params, prefix=prefix, sb_sdk2_context=context)
            for p in cls.__task_params__ if not (p.skip_in & ParamSrc.SANDBOX_SDK2_PARAM)
        }
        kwargs = {k: v for k, v in kwargs.items() if v is not MISSING}
        if common:
            kwargs.update(common)

        def dump_to_context(task_param, value):
            """
            :type task_param: TaskParamAbstract
            :type value: Any
            """
            if task_param.skip_in & ParamSrc.SANDBOX_SDK2_PARAM:
                return
            for param_name, proxy_value in task_param.proxy_params(value, params, prefix).items():
                setattr(context, param_name, proxy_value)
                setattr(
                    context,
                    CALCULATED_DEFAULT_FIELDS_NAME,
                    list(set(
                        (getattr(context, CALCULATED_DEFAULT_FIELDS_NAME, None) or []) + [param_name]
                    )),
                )
                context.save()  # noqa

        class_kwargs = kwargs
        if context:
            class_kwargs.update(__lazy_defaults_callback__=dump_to_context)

        try:
            obj = cls(**dict(class_kwargs, __src__=ParamSrc.SANDBOX_SDK2_PARAM))
        except TypeError as e:
            # TODO: remove after debugging
            logging.error('Params parsing error {}: varnames={} defaults={} kwargs={}'.format(
                e, cls.__init__.__code__.co_varnames, cls.__init__.__defaults__,
                {k: '*' * len(v) if isinstance(v, (text, bytes)) else v if 'token' in k.lower() else v for k, v in
                 kwargs.items()}
            ))
            raise

        return obj

    def proxy_sandbox_params(self, params, prefix=''):
        """
        :type params: type[sdk2.Task.Parameters]
        :type prefix: str
        """
        result = {}
        for t in self.__task_params__:
            if not (t.skip_in & ParamSrc.SANDBOX_SDK2_PARAM):
                result.update(t.proxy_params(getattr(self, t.attr_name), params, prefix=prefix))
        return result

    def to_env_args(self):
        """
        :rtype: dict[str, str]
        """
        return {
            name: value
            for name, value in (
                (env_arg_name, task_param.dump(getattr(self, task_param.attr_name), params=self))
                for env_arg_name, task_param in self.__env_args__.items()
            )
            if value
        }

    def as_dict(self):
        """
        :rtype: dict[str, str]
        """
        return {p.attr_name: getattr(self, p.attr_name) for p in self.__task_params__}

    def __repr__(self):
        return '{}({})'.format(
            self.__class__.__name__,
            ', '.join(('{}={!r}'.format(k, v) for k, v in self.as_dict().items()))
        )
