import os
import tarfile

from sandbox import sdk2
from sandbox.sandboxsdk.svn import Arcadia, Svn
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.yappy.resources import NodeJsArchive
from sandbox.projects.rpslimiter.resource_types import RpsLimiterUIBundle


def unpack(path, dest):
    tar = tarfile.open(path)
    tar.extractall(path=dest)
    tar.close()


class BuildRpsLimiterUI(sdk2.Task):
    RPS_LIMITER_UI_PATH = 'search/rpslimiter/storage/ui'

    class Parameters(sdk2.Task.Parameters):
        node_js_package = sdk2.parameters.Resource(
            "Node js package",
            required=True,
            resource_type=NodeJsArchive
        )
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)

    def on_execute(self):
        arcadia_path = Arcadia.checkout(self.Parameters.checkout_arcadia_from_url, 'arcadia', depth=Svn.Depth.IMMEDIATES)
        build_path = os.path.join(arcadia_path, self.RPS_LIMITER_UI_PATH)
        Arcadia.update(build_path, depth=Svn.Depth.INFINITY, parents=True)

        tar_path = str(sdk2.ResourceData(self.Parameters.node_js_package).path)
        node_js_path = 'nodejs'
        unpack(tar_path, node_js_path)

        node = os.path.abspath(os.path.join(node_js_path, 'bin/node'))
        npm = os.path.abspath(os.path.join(node_js_path, 'bin/npm'))
        bin_root = os.path.abspath(os.path.join(node_js_path, 'bin'))

        environment = os.environ.copy()
        environment['PATH'] = bin_root + ':' + os.environ['PATH']

        run_process(
            [node, npm, '--registry=https://npm.yandex-team.ru', 'install', '--production'],
            log_prefix='npm_install',
            work_dir=build_path,
            environment=environment
        )

        run_process(
            [node, npm, 'run', 'build_production'],
            log_prefix='npm_build_production',
            work_dir=build_path,
            environment=environment
        )

        self.create_resource(
            description='Rpslimiter ui bundle',
            resource_path=os.path.join(build_path, 'dist'),
            resource_type=RpsLimiterUIBundle,
            arch='any'
        )
