import collections
import json
import logging
import os
import yaml

from sandbox import sdk2
from sandbox.projects.common import vcs


AWACS_URL = "https://awacs.yandex-team.ru/api/"


class UpdateRpsLimiterAlerts(sdk2.Task):

    def _find_rps_limiter_installation(self, obj):
        if isinstance(obj, list):
            for l in obj:
                rps_limiter_installation = self._find_rps_limiter_installation(l)
                if rps_limiter_installation:
                    return rps_limiter_installation
        if isinstance(obj, dict):
            for k, v in obj.items():
                if k == "rps_limiter_macro":
                    installation = v.get("installation", None)
                    if not installation:
                        logging.debug("rps_limiter_macro without installation found: now it is incorrect configuration,"
                                      "but in future in will mean COMMON")
                    else:
                        return installation
                elif k == "rps_limiter":
                    external = v.get("external", None)
                    if external:
                        installation = external.get("installation", None)
                        if not installation:
                            logging.debug(
                                "easy-mode rps_limiter without installation found: now it is incorrect configuration,"
                                "but in future in will mean COMMON")
                        else:
                            return installation
                else:
                    rps_limiter_installation = self._find_rps_limiter_installation(v)
                    if rps_limiter_installation:
                        return rps_limiter_installation
        return None

    def _get_rps_limiter_installation_from_namespace(self, client, namespace_id):
        from infra.awacs.proto import api_stub, api_pb2

        balancers_stub = api_stub.BalancerServiceStub(client)
        req_pb = api_pb2.ListBalancersRequest()
        req_pb.namespace_id = namespace_id
        response_pb = balancers_stub.list_balancers(req_pb)
        for b in response_pb.balancers:
            yaml_processed = b.spec.yandex_balancer.yaml.replace("!", "")
            config_obj = yaml.safe_load(yaml_processed)
            installation = self._find_rps_limiter_installation(config_obj)
            if installation:
                return installation

        upstreams_stub = api_stub.UpstreamServiceStub(client)
        req_pb = api_pb2.ListUpstreamsRequest()
        req_pb.namespace_id = namespace_id
        response_pb = upstreams_stub.list_upstreams(req_pb)
        for u in response_pb.upstreams:
            yaml_processed = u.spec.yandex_balancer.yaml.replace("!", "")
            config_obj = yaml.safe_load(yaml_processed)
            installation = self._find_rps_limiter_installation(config_obj)
            if installation:
                return installation

        return None

    def on_execute(self):
        from infra.awacs.proto import api_stub, api_pb2
        from library.python import resource
        from nanny_rpc_client import RequestsRpcClient
        import jinja2

        self.awacs_token = sdk2.yav.Secret('sec-01djb5rpqvc0xqpg4jcnj829q1').data()['AWACS_TOKEN']
        client = RequestsRpcClient(AWACS_URL,
                                   request_timeout=20,
                                   oauth_token=self.awacs_token)
        installation_to_prj = collections.defaultdict(list)

        namespace_stub = api_stub.NamespaceServiceStub(client)
        req_pb = api_pb2.ListNamespaceSummariesRequest()

        response = namespace_stub.list_namespace_summaries(req_pb)
        for s in response.summaries:
            logging.debug("Processing namespace: " + s.id)
            installation = self._get_rps_limiter_installation_from_namespace(client, s.id)
            if installation:
                logging.debug("Found installation: " + installation)
                namespace_stub = api_stub.NamespaceServiceStub(client)
                req_pb = api_pb2.GetNamespaceRequest()
                req_pb.id = s.id
                response_pb = namespace_stub.get_namespace(req_pb)
                prj = response_pb.namespace.spec.balancer_constraints.instance_tags.prj
                if prj:
                    logging.debug('prj will be used in alert: ' + prj)
                    installation_to_prj[installation].append(prj)
            else:
                logging.debug("No installations found")
        logging.info('Installation-to-prj mapping:')
        logging.info(json.dumps(installation_to_prj, indent=4))

        installations = [
            {
                "name": installation_name,
                "prjs": ",\\\n     ".join(installation_to_prj[installation_name])
            }
            for installation_name in installation_to_prj
        ]

        template = jinja2.Template(resource.find("fails-panel-tmpl.yaml.jinja2").decode("utf-8"))
        rendered_panel = template.render(installations=installations)
        logging.info(rendered_panel)
        arc = vcs.arc.Arc(arc_oauth_token=sdk2.yav.Secret('sec-01djb5rpqvc0xqpg4jcnj829q1').data()['ARC_VCS_CLIENT_OAUTH_TOKEN'])

        with arc.mount_path(None, None, fetch_all=False) as arc_path:
            branch_name = "auto_update_rps_limiter_alerts_" + str(self.id)
            arc.checkout(arc_path, branch=branch_name, create_branch=True)
            fails_panel_path = os.path.join(arc_path, "search/tools/woland/panels/rpslimiter/fails.yaml")
            with open(fails_panel_path, "w") as fails_panel_file:
                fails_panel_file.write(rendered_panel)
            arc.add(arc_path, fails_panel_path)
            commit_message = "Auto-update RPS Limiter alerts, task id = " + str(self.id)
            arc.commit(arc_path, message=commit_message)
            arc.push(arc_path, upstream="users/robot-rpslimiter/" + branch_name)
            pr_create_output = arc.pr_create(arc_path, message=commit_message, publish=True, auto=True)
            logging.info(pr_create_output)
