import os
import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn
from sandbox.sdk2.helpers import subprocess


bad_events = [
    ctt.Status.FAILURE,
    ctt.Status.Group.BREAK  # exception, timeout, stopped, expired, no_res
]


class RtcSaltTests(sdk2.Task):
    GIT_URL = 'https://x-oauth-token:{}@bb.yandex-team.ru/scm/rtcsalt/saltstack.git'

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600
        ref_id = sdk2.parameters.String("ref_id", required=True)
        ref_sha = sdk2.parameters.String("ref_sha", required=False)
        release = sdk2.parameters.Bool("release", required=False, default=False)
        notifications = [
            sdk2.Notification(
                bad_events,
                ['Reddoggad'],
                ctn.Transport.TELEGRAM
            )
        ]

    def _prepare_venv(self):
        """
        Prepare virtual env
        """
        logging.debug('Starting prepare virtualenv')
        venv_path = os.path.join(os.getcwd(), '_venv')
        resource = sdk2.Resource["VIRTUALENV"].find(
            attrs=dict(target='RTC_SALT_TEST')).first()
        path_venv_tar = str(sdk2.ResourceData(resource).path)
        # Create directory
        subprocess.check_call(['mkdir', '-p', venv_path])
        # Extract venv
        subprocess.check_call(['tar', '-zxf', path_venv_tar, '-C', venv_path])
        return venv_path

    def _prepare_git(self, url, path, ref, commit=None):
        """
        Prepare Git repository (clone, checkout)
        """
        if 'refs/heads/' in ref:
            ref = ref[len('refs/heads/'):]
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('git')) as pl:
            # Clone project
            subprocess.check_call(
                ['git', 'clone', '--branch', ref, url, path],
                stdout=pl.stdout, stderr=subprocess.STDOUT)
            # Switch to commit id if needed
            if commit:
                subprocess.check_call(
                    ['git', 'checkout', commit], cwd=path,
                    stdout=pl.stdout, stderr=subprocess.STDOUT)
        return os.path.join(path)

    def _run_tests(self, path, pytest_path):
        """
        Run pytest
        """
        with sdk2.helpers.ProcessLog(
                self,
                logger=logging.getLogger('pytest')) as pl:
            # Run pytest
            subprocess.check_call(
                [pytest_path, 'tests'], stdout=pl.stdout,
                stderr=subprocess.STDOUT,
                cwd=path)

    def on_execute(self):
        """
        Main
        """
        venv_path = self._prepare_venv()
        pytest_path = os.path.join(venv_path, 'bin', 'pytest')
        repo_path = os.path.join(os.getcwd(), 'saltstack')
        logging.info("Starting ... ")
        logging.info('Repo url : {}'.format(self.GIT_URL))
        logging.info('Repo path : {}'.format(repo_path))
        logging.info('Ref id : {}'.format(self.Parameters.ref_id))
        logging.info('Ref sha : {}'.format(self.Parameters.ref_sha))
        logging.info('Venv Path : {}'.format(venv_path))
        logging.info('Pytest Path : {}'.format(pytest_path))
        self._prepare_git(url=self.GIT_URL.format(
            sdk2.Vault.data('RTC_SALT_GIT_TOKEN')),
            path=repo_path,
            ref=self.Parameters.ref_id,
            commit=self.Parameters.ref_sha)
        self._run_tests(path=repo_path, pytest_path=pytest_path)
