import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.nanny import nanny


class RtcSlaTentaclesBackendConfigTgz(sdk2.Resource):
    """ Tgz archive file with RTC SLA Tentacles backend configs inside. """
    any_arch = True
    auto_backup = False
    executable = False
    releasable = True
    ttl = 28


class RtcSlaTentaclesBackendConfig(nanny.ReleaseToNannyTask2, sdk2.Task):
    """ Generates tgz archive file with RTC SLA Tentacles backend configs inside. """

    class Context(sdk2.Task.Context):
        tentacles_backend_arcadia_root = os.path.join("infra", "rtc_sla_tentacles", "backend")
        tentacles_backend_config_dir = "conf"
        resource_archive_name = "RtcSlaTentaclesBackendConfigTgz.tgz"
        resource_description_prefix = "RTC SLA Tentacles Backend - Config"

    class Parameters(sdk2.Parameters):
        arcadia_commit_number = sdk2.parameters.Integer("Arcadia commit number ('0' means trunk)",
                                                        required=True,
                                                        default_value=0)
        arc_review_number = sdk2.parameters.Integer("Arc review number ('0' means trunk)",
                                                    required=False,
                                                    default_value=0)
        resource_comment = sdk2.parameters.String("Resource changes comment",
                                                  required=False,
                                                  default_value="")

    def svn_export(self, tentacles_backend_arcadia_root):
        arcadia_commit_number = self.Parameters.arcadia_commit_number
        arc_review_number = self.Parameters.arc_review_number

        if arcadia_commit_number != 0:
            arcadia_url = "arcadia:/arc/trunk/arcadia/{}@{}".format(tentacles_backend_arcadia_root,
                                                                    str(arcadia_commit_number))
        else:
            arcadia_url = "arcadia:/arc/trunk/arcadia/{}".format(tentacles_backend_arcadia_root)

        sdk2.vcs.svn.Arcadia.export(arcadia_url, tentacles_backend_arcadia_root)

        if arc_review_number != 0:
            arc_review_name = "arc:{}".format(str(arc_review_number))
            sdk2.vcs.svn.Arcadia.apply_patch(str(self.path()),
                                             arc_review_name,
                                             str(self.path()))

    def on_execute(self):
        tentacles_backend_arcadia_root = self.Context.tentacles_backend_arcadia_root
        tentacles_backend_config_dir = self.Context.tentacles_backend_config_dir
        resource_archive_name = self.Context.resource_archive_name
        resource_description_prefix = self.Context.resource_description_prefix
        resource_comment = self.Parameters.resource_comment

        self.svn_export(tentacles_backend_arcadia_root)

        local_path_tentacles_root = os.path.join(str(self.path()),
                                                 tentacles_backend_arcadia_root)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("tree")) as pl:
            sp.Popen("tree {}".format(tentacles_backend_config_dir),
                     shell=True,
                     cwd=local_path_tentacles_root,
                     stdout=pl.stdout,
                     stderr=sp.STDOUT).wait()
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("tar")) as pl:
            sp.Popen("tar -cf RtcSlaTentaclesBackendConfigTgz.tgz {}".format(tentacles_backend_config_dir),
                     shell=True,
                     cwd=local_path_tentacles_root,
                     stdout=pl.stdout,
                     stderr=sp.STDOUT).wait()

        if resource_comment:
            resource_description = "{}: {}".format(resource_description_prefix,
                                                   resource_comment)
        else:
            resource_description = resource_description_prefix

        path_to_archive = os.path.join(local_path_tentacles_root,
                                       resource_archive_name)

        sdk2.ResourceData(RtcSlaTentaclesBackendConfigTgz(self,
                                                          resource_description,
                                                          path_to_archive))

    # Set ttl=inf on released resource.
    def mark_released_resources(self, status, ttl="inf"):
        super(RtcSlaTentaclesBackendConfig, self).mark_released_resources(status, ttl)

    def on_release(self, additional_parameters):
        additional_parameters['release_comments'] += 'RTC SLA Tentacles release\n'
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
