# -*- coding: utf-8 -*-

import os
import sys
import logging
from pprint import pformat
from contextlib import contextmanager

from sandbox import sdk2
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.environments import VirtualEnvironment

from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.yappy.utils.clickhouse import SetRequirementsFromClickhouseMixin

trunk_url = str(sdk2.svn.Arcadia.trunk_url())

_logger = logging.getLogger('{}_{}'.format('BaseRtccProject', __name__))


class ArcadiaSdk2Task(sdk2.Task):
    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.String(
            'Svn url for arcadia', default=trunk_url, required=True)
        arcadia_patch = sdk2.parameters.String(
            'Patch for arcadia'
        )

    def get_arcadia_src_dir(self, ignore_externals=False, copy_trunk=False):
        arcadia_src_dir = sdk2.svn.Arcadia.get_arcadia_src_dir(
            self.Parameters.checkout_arcadia_from_url, copy_trunk=copy_trunk)
        if not arcadia_src_dir:
            raise errors.SandboxTaskFailureError(
                'Cannot get repo for url {0}'.format(self.Parameters.checkout_arcadia_from_url)
            )
        return arcadia_src_dir

    def apply_patch(self, arcadia_src_dir, dest_dir):
        sdk2.svn.Arcadia.apply_patch(arcadia_src_dir, self.Parameters.arcadia_patch, dest_dir)

    def apply_to_root(self, arcadia_src_dir):
        self.apply_patch(arcadia_src_dir, str(self.path()))
        return arcadia_src_dir


class ArcadiaRtccTask(ArcadiaSdk2Task):
    class Parameters(ArcadiaSdk2Task.Parameters):
        rtcc_path = sdk2.parameters.String(
            'Path to rtcc src relatively to arcadia trunk root',
            default='search/priemka/gencfg.upper',
            required=True
        )

    class Context(sdk2.Context):
        rtcc_path = None

    def get_rtcc_path(self):
        return os.path.join(self.apply_to_root(self.get_arcadia_src_dir()), self.Parameters.rtcc_path)

    def _process_requirements(self, requirements_path):
        """
        Collects python package requirements and returns them at sandbox VirtualEnvironment compatible format
        :param requirements_path: path to source requirements.txt file
        :type requirements_path: str
        :return: List of package requirements (with versions)
        :rtype: list
        """
        requirements = []

        for line in [ln.strip() for ln in open(requirements_path).readlines()]:
            if line.startswith('-r'):
                embed_req_path = os.path.abspath(os.path.join(
                    os.path.dirname(requirements_path),
                    line.split(' ')[-1].strip()
                ))
                _logger.info('<Arcadia Rtcc> Processing embedded requirements file: {}'.format(embed_req_path))
                requirements += self._process_requirements(embed_req_path)
            elif line.startswith('-'):
                # ignore --index-uri
                pass
            else:
                requirements.append(line)

        _logger.info('<Arcadia Rtcc> Requirements from {} are: {}'.format(requirements_path, requirements))

        return requirements

    @contextmanager
    def venv_with_rtcc_reqs(self, requirements_path=None):
        """
        Acts as a wrapper over sandbox VirtualEnviroment (used as contextmanager). Additionally collects and
        installs package requirements.
        :param requirements_path: path to source requirements.txt file
        :type requirements_path: Union[str, tuple, list]
        """
        requirements_path = os.path.join(self.get_rtcc_path(),
                                         *(requirements_path if requirements_path else ('requirements.txt',)))
        with VirtualEnvironment(use_system=True) as venv:
            venv.pip(' '.join(self._process_requirements(requirements_path)))
            yield

    @staticmethod
    def _log_tracebacks(call, call_args, call_kwargs, exc_type):
        try:
            return call(*call_args, **call_kwargs)
        except exc_type as e:
            _logger.info('cwd contents %s', os.popen('ls -la').read())
            t_back = sys.exc_info()[2]
            while t_back.tb_next:
                t_back = t_back.tb_next
            _logger.info('Caught exception: {}'.format(e))
            _logger.info('Exception innermost frame locals: {}'.format(pformat(t_back.tb_frame.f_locals)))
            raise

    @property
    def rtcc_path(self):
        if self.Context.rtcc_path is None:
            arcadia_src = self.Parameters.checkout_arcadia_from_url
            arcadia_src_dir = self._log_tracebacks(
                sdk.do_clone,
                (arcadia_src, self),
                {'use_checkout': False},
                OSError
            )

            release_svn_info = sdk2.svn.Arcadia.info(arcadia_src)
            _logger.info('<CLONING RTCC> Release revision is %s.', release_svn_info['entry_revision'])
            sdk2.svn.Arcadia.update(
                os.path.join(arcadia_src_dir, 'search/priemka'),
                revision=release_svn_info['entry_revision'],
                set_depth='infinity',
                parents=True
            )
            self.Context.rtcc_path = os.path.join(self.apply_to_root(arcadia_src_dir), self.Parameters.rtcc_path)

        _logger.info('RTCC path is: %s', self.Context.rtcc_path)
        return self.Context.rtcc_path


class DynamicPlaceTask(SetRequirementsFromClickhouseMixin, ArcadiaRtccTask):

    def on_enqueue(self):
        self.set_requirements()
