# -*- coding: utf-8 -*-
import os
import shutil
import logging

import requests

from sandbox import sdk2
import sandbox.common.types.resource as ct_resource
import sandbox.common.types.client as ctc
import sandbox.sdk2.helpers as s2h
from sandbox.sandboxsdk import environments as pyenv
from sandbox.sandboxsdk.channel import channel

import sandbox.projects.common.arcadia.sdk as sdk
from sandbox.projects.common import error_handlers as eh
from sandbox.projects import resource_types as rt

from sandbox.projects.rtcc import resources as rtccres
from sandbox.projects.rtcc.base import mixin as rtccmixin

from sandbox.projects.release_machine.components.configs.rtcc import RtccCfg
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import rm_notify as rm_notify
from sandbox.projects.common.sdk_compat import task_helper

from sandbox.projects.websearch.upper import resources as upper_resources


NANNY_ADDR = 'https://nanny.yandex-team.ru'


@rm_notify.notify2()
class BuildRtccDiff(rtccmixin.DynamicPlaceTask):
    """
    Generates diffs from two of rtcc bundle generation processes
    """

    DIFF_FULL_FILENAME = 'config.upper.bundle.diff.full.txt'
    DIFF_BRIEF_FILENAME = 'config.upper.bundle.diff.brief.txt'

    CACHE_DIRNAME = 'cache'
    DIFF_SIGNAL_FILENAME = 'has_diff.txt'

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.GENERIC

    class Parameters(rtccmixin.ArcadiaSdk2Task.Parameters):

        rtcc_path = sdk2.parameters.String(
            'Path to rtcc src relatively to arcadia trunk root',
            default='search/priemka/gencfg.upper',
            required=True
        )

        with sdk2.parameters.Group('Compare with production'):
            compare_with_prod = sdk2.parameters.Bool(
                'Compare "NEW" RTCC_BUNDLE with production resource (instead of last released)',
                default=False
            )
            referent_nanny_rtcc_bundle_carrier = sdk2.parameters.String(
                'Service from where to get "OLD" RTCC_BUNDLE',
            )
            nanny_token_owner = sdk2.parameters.String(
                'Owner of nanny token',
                default='MAESTRO'
            )
            nanny_token_name = sdk2.parameters.String(
                'Nanny token name at sandbox vault',
                default='maestro_dummy_service__nanny_oauth_token'
            )

        with sdk2.parameters.Group('RTCC bundles and caches'):
            with sdk2.parameters.RadioGroup('Contour type') as ctype:
                ctype.values['priemka'] = ctype.Value(value='priemka')
                ctype.values['production'] = ctype.Value(value='production', default=True)
                ctype.values['production_noapache'] = ctype.Value(value='noapache production')

            with ctype.value['priemka']:
                priemka_old_bundle = sdk2.parameters.LastReleasedResource(
                    'Old RTCC_BUNDLE_PRIEMKA', resource_type=rt.RTCC_BUNDLE_PRIEMKA, state=(ct_resource.State.READY,))
                priemka_new_bundle = sdk2.parameters.Resource(
                    'New RTCC_BUNDLE_PRIEMKA', resource_type=rt.RTCC_BUNDLE_PRIEMKA, state=(ct_resource.State.READY,))
                priemka_old_cache = sdk2.parameters.LastReleasedResource(
                    'Old RTCC_CACHE', resource_type=rt.RTCC_CACHE, state=(ct_resource.State.READY,))
                priemka_new_cache = sdk2.parameters.Resource(
                    'New RTCC_CACHE', resource_type=rt.RTCC_CACHE, state=(ct_resource.State.READY,))

            with ctype.value['production']:
                production_old_bundle = sdk2.parameters.LastReleasedResource(
                    'Old RTCC_BUNDLE', resource_type=rt.RTCC_BUNDLE, state=(ct_resource.State.READY,))
                production_new_bundle = sdk2.parameters.Resource(
                    'New RTCC_BUNDLE', resource_type=rt.RTCC_BUNDLE, state=(ct_resource.State.READY,))
                production_old_cache = sdk2.parameters.LastReleasedResource(
                    'Old RTCC_CACHE', resource_type=rt.RTCC_CACHE, state=(ct_resource.State.READY,))
                production_new_cache = sdk2.parameters.Resource(
                    'New RTCC_CACHE', resource_type=rt.RTCC_CACHE, state=(ct_resource.State.READY,))

            with ctype.value['production_noapache']:
                production_noapache_old_bundle = sdk2.parameters.LastReleasedResource(
                    'Old RTCC_BUNDLE_NOAPACHE', resource_type=upper_resources.RtccBundleNoapache, state=(ct_resource.State.READY,))
                production_noapache_new_bundle = sdk2.parameters.Resource(
                    'New RTCC_BUNDLE_NOAPACHE', resource_type=upper_resources.RtccBundleNoapache, state=(ct_resource.State.READY,))
                production_noapache_old_cache = sdk2.parameters.LastReleasedResource(
                    'Old RTCC_CACHE_NOAPACHE', resource_type=upper_resources.RtccCacheNoapache, state=(ct_resource.State.READY,))
                production_noapache_new_cache = sdk2.parameters.Resource(
                    'New RTCC_CACHE_NOAPACHE', resource_type=upper_resources.RtccCacheNoapache, state=(ct_resource.State.READY,))

    class Context(sdk2.Context):
        brief_diff = 'NOT GENERATED YET'
        has_diff = False
        marty_release_info = None
        prod_bundle_path = None
        prod_cache_path = None

    # SEARCHPIEMKA-982
    def _extract_rtcc_cache_from_task(self, task_id):
        try:
            return channel.sandbox.list_resources(resource_type='RTCC_CACHE', task_id=task_id)[0]
        except IndexError:
            eh.check_failed('Unable to get RTCC_CACHE from task {}'.format(task_id))

    def _get_rtcc_bundle_id_from_nanny(self):
        token = sdk2.Vault.data(self.Parameters.nanny_token_owner, self.Parameters.nanny_token_name)
        # FIXME(mvel): retries, use nanny client?
        r = requests.get(
            '{0}/v2/services/{1}/runtime_attrs/'.format(NANNY_ADDR, self.Parameters.referent_nanny_rtcc_bundle_carrier),
            headers={'Accept': 'application/json', 'Authorization': 'OAuth {}'.format(token)}
        )

        try:
            for resource in r.json()["content"]["resources"]["sandbox_files"]:
                if resource['resource_type'] == self.Context.bundle_resource_type_name:
                    return resource['task_id']
            else:
                eh.check_failed('Referent service {} has no {}'.format(
                    self.Parameters.referent_nanny_rtcc_bundle_carrier, self.Context.bundle_resource_type_name,
                ))
        except KeyError:
            eh.check_failed('Unable to get {} info from nanny (maybe something with token?)'.format(
                self.Parameters.referent_nanny_rtcc_bundle_carrier
            ))

    def _get_production_bundle(self):
        rtcc_bundle_task_id = self._get_rtcc_bundle_id_from_nanny()
        eh.ensure(
            rtcc_bundle_task_id,
            'Unable to get {} from {}'.format(
                self.Context.bundle_resource_type_name, self.Parameters.referent_nanny_rtcc_bundle_carrier
            )
        )
        bundle_resources = channel.sandbox.list_resources(
            task_id=rtcc_bundle_task_id, resource_type=self.Context.bundle_resource_type_name
        )
        eh.ensure(bundle_resources, 'Unable to get {} from task {}'.format(self.Context.bundle_resource_type_name, rtcc_bundle_task_id))
        return bundle_resources[0]

    def get_production_bundle_and_cache(self):
        bundle = self._get_production_bundle()
        cache = self._extract_rtcc_cache_from_task(bundle.task_id)
        self.Context.prod_bundle_path = str(sdk2.ResourceData(bundle).path)
        self.Context.prod_cache_path = str(sdk2.ResourceData(cache).path)

    def _get_diff_path(self, diff_type):
        """
        Return file path for a given diff type
        :param diff_type: diff type (`full`, 'first_1000` or `brief`)
        :type diff_type: str
        :return: path to location where diff file must reside
        :rtype: str
        """
        return str(self.path(getattr(self, 'DIFF_{}_FILENAME'.format(diff_type.upper()))))

    def _create_resource_data(self, resource_type, data_path):
        """
        Create resource and mark this resource as ready
        :param resource_type: Sandbox resource type
        :type resource_type: type
        :param data_path: Where to set resource data path
        :type data_path: str
        """
        resource = resource_type(self, 'Diff data from task {}'.format(self.id), path=data_path)
        sdk2.ResourceData(resource).ready()
        return resource.id

    def _prepare_inbound_resources(self, resource_kind):
        """
        Syncronizes resource pair (old and new) on disk and returns paths to them
        :param resource_kind: type of resource, can be 'cache' or 'bundle'
        :type resource_kind: str
        :return: two-element tuple containing resource paths
        :rtype: tuple
        """
        if self.Parameters.compare_with_prod:
            _param = '{ctype}_new_{kind}'.format(ctype=self.Parameters.ctype, kind=resource_kind)
            if resource_kind == 'bundle':
                return (str(sdk2.ResourceData(getattr(self.Parameters, _param).path)), self.Context.prod_bundle_path)
            else:
                return (str(sdk2.ResourceData(getattr(self.Parameters, _param).path)), self.Context.prod_cache_path)

        resource_params = ['{ctype}_{freshness}_{kind}'.format(
            ctype=self.Parameters.ctype, freshness=freshness, kind=resource_kind
        ) for freshness in ('old', 'new')]

        return tuple(sdk2.ResourceData(getattr(self.Parameters, res_param)).path for res_param in resource_params)

    def _resource_from_subproc_out(self, diff_type, arguments, resource_class):
        """
        Runs resource generation process, captures resource data and creates appropriate Sandbox resource
        :param diff_type: type of diff to generate
        :type diff_type: str
        :param arguments: arguments to subprocess.Popen
        :type arguments: Union[str, list]
        :param resource_class: Resource class which instance will be created
        :type resource_class: type
        """
        with s2h.ProcessLog(self, 'rtcc_diff') as pl, open(self._get_diff_path(diff_type), 'w+') as diff_fd:
            s2h.subprocess.Popen(arguments, stdout=diff_fd, stderr=pl.stderr).wait()

        self._create_resource_data(resource_class, self._get_diff_path(diff_type))

    def make_full_diff(self):
        """
        Create full diff between resources (with standard POSIX `diff` utility`
        """
        old_bundle_path, new_bundle_path = self._prepare_inbound_resources('bundle')
        logging.info('Launching creation of full diff between {!s} and {!s}'.format(old_bundle_path, new_bundle_path))
        diff_arguments = [
            'diff',
            '-ENwbur',
            str(old_bundle_path),
            str(new_bundle_path),
            '--ignore-matching-lines=^#.*',
            '--ignore-matching-lines=^ConfigVersion.*'
        ]
        self._resource_from_subproc_out('full', diff_arguments, rtccres.RtccFullDiff)

    def make_brief_diff(self):
        """
        Creates `brief` diff between two RTCC_CACHE resources
        """
        old_cache_path, new_cache_path = map(str, self._prepare_inbound_resources('cache'))

        logging.info('Resource path for old cache: %s. -"- for new cache: %', old_cache_path, new_cache_path)

        # In local installation `chmod` fails (as expected), so copying new_cache into current task's dir
        # Rtcc also manipulates with old cache file tree (session operations), so we must locally sync
        # old cache resource too
        local_new_cache_path = str(self.path('new_rtcc_cache'))
        local_old_cache_path = str(self.path('old_rtcc_cache'))

        shutil.copytree(new_cache_path, local_new_cache_path)
        shutil.copytree(old_cache_path, local_old_cache_path)

        for path in (local_new_cache_path, local_old_cache_path):
            s2h.subprocess.Popen(['chmod', '-R', '+w', path]).wait()

        arcadia_src = self.Parameters.checkout_arcadia_from_url
        arcadia_src_dir = sdk.do_clone(arcadia_src, self, use_checkout=False)
        release_svn_info = sdk2.svn.Arcadia.info(arcadia_src)
        logging.info('Release revision is %s.', release_svn_info['entry_revision'])
        sdk2.svn.Arcadia.update(
            os.path.join(arcadia_src_dir, 'search/priemka'),
            revision=release_svn_info['entry_revision'],
            set_depth='infinity',
            parents=True
        )

        rtcc_path = os.path.join(self.apply_to_root(arcadia_src_dir), self.Parameters.rtcc_path)
        venv = pyenv.VirtualEnvironment(use_system=True)
        requirements_path = os.path.join(rtcc_path, 'requirements.txt')
        req_specs = " ".join([line.strip() for line in open(requirements_path).readlines()])

        with venv:
            pyenv.PipEnvironment('pip', version="9.0.1", venv=venv).prepare()
            pyenv.PipEnvironment('setuptools', version="39.2.0", venv=venv).prepare()
            pyenv.PipEnvironment('setuptools-scm', version="5.0.2", venv=venv).prepare()
            try:
                venv.pip(req_specs)
            except pyenv.PipError:
                venv.pip(req_specs)

            arguments = [
                'python',
                os.path.join(rtcc_path, 'rtcc.py'),
                'diff',
                local_old_cache_path,
                local_new_cache_path,
                '--report={}'.format(self._get_diff_path('brief')),
                '--mark_diff',
            ]

            with s2h.ProcessLog(self, 'brief_rtcc_diff') as pl:
                # There are no need to check return status since status depends on diff existence only
                s2h.subprocess.Popen(arguments, stdout=pl.stdout, stderr=pl.stderr).wait()

            # Checks if diff file have been generated
            assert os.path.exists(self._get_diff_path('brief')), 'Brief diff was not generated'
            brief_diff_id = self._create_resource_data(rtccres.RtccBriefDiff, self._get_diff_path('brief'))
            # Brief diff sets here for displaying at task's page footer
            self.Context.brief_diff = open(self._get_diff_path('brief')).read()
            self.set_info(
                'Brief diff resource created: https://sandbox.yandex-team.ru/resource/{}/view'.format(brief_diff_id)
            )

    def has_diff(self):
        """
        Checks whether diff exists between two config generator runs
        :rtype: int
        """
        diff_file_path = str(self.path(self.DIFF_SIGNAL_FILENAME))
        result = 0

        try:
            with open(diff_file_path, 'r') as dsf:
                result = int(dsf.read().strip())  # Because it looks easy to do with it in release cycle
            os.unlink(diff_file_path)  # Preventing false positives on task restarts etc.
        except IOError as e:
            logging.info('Error occured during operations with diff `signal` file. Original exception was {}'.format(e))
            # FIXME: raise task failure here

        return result

    def extract_marty_release_info(self):
        """
        Extracts release notes (incl. info about mmeta reclustering) generated at diff building time
        """
        marty_info_path = str(self.path('new_rtcc_cache', 'reports', 'release.notes.marty.txt'))
        try:
            with open(marty_info_path) as info_fp:
                self.Context.marty_release_info = info_fp.read()
        except IOError as e:
            logging.info(
                'Error occured during operations with release info for marty. Original exception was {}'.format(e))
            # FIXME: raise task failure here

    def on_enqueue(self):
        task_helper.ctx_field_set(self, rm_const.COMPONENT_CTX_KEY, RtccCfg.name)
        self.Context.bundle_resource_type_name = str(getattr(self.Parameters, '{ctype}_new_bundle'.format(ctype=self.Parameters.ctype)).type)
        rtccmixin.DynamicPlaceTask.on_enqueue(self)

    def on_execute(self):
        if self.Parameters.compare_with_prod:
            self.get_production_bundle_and_cache()

        self.make_full_diff()
        self.make_brief_diff()
        self.Context.has_diff = self.has_diff()
        self.extract_marty_release_info()

    @property
    def footer(self):
        """
        Displays full `brief' diff here (as in the TeamCity)
        :return: Full 'brief' diff contents
        :rtype: str
        """
        return self.Context.brief_diff.replace('\n', '<br>')
