# -*- coding: utf-8 -*-
import json
import pprint
import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt


_BACKEND_PARAM_DESCRIPTION = \
    'Mapping from source names to inherit from to ;-delimited sequence of new source name and new backend group' \
    '(sequence can be a json-encoded list of such sequences and many groups can be comma-separated)'

_KNOWN_LOCATIONS = ['MAN', 'SAS', 'VLA']

_DEFAULT_BACKENDS = {
    'report': {
        'AH_HTTP_ADAPTER': [
            'AH_HTTP_ADAPTER_YAPPY_STABLE;' + ','.join(['{}_NEWS_ACCEPTANCE_STABLE_AH_HTTP_ADAPTER'.format(l) for l in _KNOWN_LOCATIONS]),
            'AH_HTTP_ADAPTER_YAPPY_TESTING;' + ','.join(['{}_NEWS_ACCEPTANCE_TESTING_AH_HTTP_ADAPTER'.format(l) for l in _KNOWN_LOCATIONS])
        ]
    },
    'noapache': {
        'NEWSD_RUSSIAN': [
            'NEWSD_RUSSIAN_YAPPY_STABLE;' + ','.join(['{}_NEWS_ACCEPTANCE_STABLE_SLAVE_NEWSD'.format(l) for l in _KNOWN_LOCATIONS]),
            'NEWSD_RUSSIAN_YAPPY_TESTING;' + ','.join(['{}_NEWS_ACCEPTANCE_TESTING_SLAVE_NEWSD'.format(l) for l in _KNOWN_LOCATIONS])
        ],
    }
}


class BuildConfigForNewsBetas(sdk2.Task):
    """
    Patch RTCC with yappy betas backends
    """

    class Parameters(sdk2.Parameters):
        patch_contour_name = sdk2.parameters.String('Source contour name', default='yappy_beta', required=True)
        project = sdk2.parameters.String('Source project', default='news', required=True)
        location = sdk2.parameters.String('Source location(==dc)', default='man', required=True)
        domain = sdk2.parameters.String('Source domain (always rkub)', default='rkub', required=True)
        source_contour = sdk2.parameters.String('Source contour (production/hamster)', default='production', required=True)
        report_backends = sdk2.parameters.Dict(_BACKEND_PARAM_DESCRIPTION, required=False)
        noapache_backends = sdk2.parameters.Dict(_BACKEND_PARAM_DESCRIPTION, required=False)

    def get_rtcc_instances(self, instance_type, config_patch,
                           patch_type=None, stand_name=None):
        patch_type = patch_type or instance_type
        instance_full_type = '{}_{}'.format(instance_type, self.Parameters.project)
        stand_name = stand_name or '{}_{}'.format(instance_full_type, self.Parameters.domain)
        return {
            'config': {
                'parent': {
                    'contour': self.Parameters.source_contour,
                    'location': self.Parameters.location,
                    'stand_name': stand_name,
                },
                'patch': config_patch
            },
            'type': patch_type,
            'stand_name': stand_name,
            'location': self.Parameters.location,
        }

    def _get_patch(self, patch_input_data, instance_type, patch_type=None):
        instances = []

        if isinstance(patch_input_data, (str, unicode)):
            try:
                patch_input_data = json.loads(patch_input_data)
            except ValueError:
                pass

        patch = {}
        for inherit_from, patch_data in patch_input_data.items():
            if not isinstance(patch_data, (list, tuple)):
                patch_data = [patch_data]
            for patch_data_item in patch_data:
                source_name, source_groups = patch_data_item.split(';')
                patch[source_name] = {
                    'instances': {
                        'type': 'cs_online',
                        'expression': source_groups,
                        'args': []
                    },
                    'inherit_from': inherit_from
                }
        instances.append(
            self.get_rtcc_instances(instance_type=instance_type, config_patch=patch,
                                    patch_type=patch_type))
        return instances

    def _get_noapache_patch(self):
        backends = self.Parameters.noapache_backends or _DEFAULT_BACKENDS['noapache']
        return self._get_patch(backends, instance_type='noapache')

    def _get_report_patch(self):
        backends = self.Parameters.report_backends or _DEFAULT_BACKENDS['report']
        return self._get_patch(backends, instance_type='upper', patch_type='sfront')

    def get_rtcc_patch(self):
        return [{
            'name': self.Parameters.patch_contour_name,
            'stands': self._get_noapache_patch() + self._get_report_patch()
        }]

    def run_rtcc_build(self, patch):
        BuildUpperConfig = sdk2.Task['BUILD_UPPER_CONFIG']
        subtask_id = BuildUpperConfig(
            self,
            description='Created From BuildConfigForNewsBetas #{}'.format(self.id),
            use_raw_patch=True,
            raw_patch_parameter=json.dumps(patch)
        ).enqueue()

        raise sdk2.WaitTask([subtask_id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.rtcc_build:
            patch = self.get_rtcc_patch()
            logging.info('Got rtcc patch: %s', pprint.pformat(patch))
            self.run_rtcc_build(patch)
