# -*- coding: utf-8 -*-

import os
import logging

from sandbox import sdk2
import sandbox.projects.common.arcadia.sdk as sdk
import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.environments as pyenv
import sandbox.sandboxsdk.errors as sb_errors
from sandbox.sdk2.helpers import subprocess as sb_subprocess

from sandbox.projects.rtcc.base.mixin import (
    ArcadiaSdk2Task,
    DynamicPlaceTask
)

from sandbox.projects.release_machine.components.configs.rtcc import RtccCfg
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import rm_notify as rm_notify
from sandbox.projects.common.sdk_compat import task_helper


@rm_notify.notify2()
class RunRtccUnitTests(DynamicPlaceTask):

    class Requirements(sdk2.Requirements):
        disk_space = 5000  # 5G
        client_tags = ctc.Tag.GENERIC

    class Parameters(ArcadiaSdk2Task.Parameters):
        rtcc_path = sdk2.parameters.String(
            'Path to rtcc src relatively to arcadia trunk root',
            default='search/priemka/gencfg.upper',
            required=True
        )

    class Context(sdk2.Context):
        footer_content = {}
        rtcc_path = None

    def _run_tests(self):
        tests_path = os.path.join(self.Context.rtcc_path, 'tests')
        logging.info('Running unit tests from {}'.format(tests_path))
        proc = sb_subprocess.Popen(
            'python -m pytest -m "not long" {}'.format(tests_path),
            shell=True,
            stdout=sb_subprocess.PIPE,
            stderr=sb_subprocess.PIPE
        )
        out, err = proc.communicate()
        self.Context.footer_content = {
            'out': out.replace('\n', '<br>'),
            'err': err.replace('\n', '<br>')
        }
        return proc.returncode

    def install_and_run(self):
        with pyenv.VirtualEnvironment(use_system=True) as venv:
            req_specs = " ".join(
                [line.strip() for line in open(os.path.join(self.Context.rtcc_path, 'requirements.txt')).readlines()])
            pyenv.PipEnvironment('setuptools', version="39.2.0", venv=venv).prepare()
            venv.pip(req_specs)
            os.environ['PYTHONPATH'] = '{}:{}'.format(os.environ['PYTHONPATH'], self.Context.rtcc_path)
            result = self._run_tests()
        return result

    def on_execute(self):
        arcadia_src = self.Parameters.checkout_arcadia_from_url
        arcadia_src_dir = sdk.do_clone(arcadia_src, self, use_checkout=False)
        release_svn_info = sdk2.svn.Arcadia.info(arcadia_src)
        logging.info('Release revision is %s.', release_svn_info['entry_revision'])
        sdk2.svn.Arcadia.update(
            os.path.join(arcadia_src_dir, 'search', 'priemka'),
            revision=release_svn_info['entry_revision'],
            set_depth='infinity',
            parents=True
        )

        self.Context.rtcc_path = os.path.join(self.apply_to_root(arcadia_src_dir), self.Parameters.rtcc_path)
        result = self.install_and_run()
        if result != 0:
            raise sb_errors.SandboxTaskFailureError(
                'RTCC Unit test run failed with code {}. See task logs and footer for details'.format(result))

    def on_enqueue(self):
        task_helper.ctx_field_set(self, rm_const.COMPONENT_CTX_KEY, RtccCfg.name)
        DynamicPlaceTask.on_enqueue(self)

    @property
    def footer(self):
        return self.Context.footer_content or 'No data yet'
