import logging
import os
import shutil
import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
import sandbox.projects.common.constants as consts
import sandbox.projects.common.build.parameters as build_parameters

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common.build.sdk import sdk_compat as ya_sdk_compat
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.common.build import ya_make_package
from sandbox.projects.rthub.common import common

from os.path import join as pj

logger = logging.getLogger(__name__)


class BuildRthubCm(ReleaseToNannyTask2, YaMake2):

    class Context(sdk2.Task.Context):
        child_tasks = dict()

    class Requirements(sdk2.Task.Requirements):
        client_tags = Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        build_output_html_ttl = 7
        build_output_ttl = 7

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('Vcs url for arcadia', required=True)

        with sdk2.parameters.CheckGroup("Resources to build") as build_targets:
            for r in sorted(common.RTHUB_CM_TARGETS):
                build_targets.values[r] = r

        with sdk2.parameters.RadioGroup('Build type') as build_type:
            build_type.values.release = build_type.Value('release', default=True)
            build_type.values.debug = 'debug'
            build_type.values.profile = 'profile'

        arcadia_patch = sdk2.parameters.String(
            'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
            default='', multiline=True
        )

        with sdk2.parameters.RadioGroup("Sanitize") as sanitize_group:
            sanitize_group.values.none = sanitize_group.Value(default=True)
            sanitize_group.values.address = None
            sanitize_group.values.leak = None
            sanitize_group.values.memory = None
            sanitize_group.values.thread = None
            sanitize_group.values.undefined = None

        definition_flags = build_parameters.DefinitionFlags()
        yt_store_parameters = build_parameters.YtStoreParameters()

        coverage_prefix_filter = None
        coverage_exclude_regexp = ''
        coverage_unified_agent = False
        coverage_unified_agent_sid = ''
        coverage_unified_agent_strict = False
        distbuild_pool = None
        test_size_filter = None
        java_coverage = False
        javac_options = None
        jvm_args = None
        musl = False
        lto = False
        thinlto = False
        pgo_add = False
        pgo_use = None
        pgo_merge_timeout = 600
        vault_owner = None
        vault_key_name = None
        test_params = {}
        clear_build = False
        keep_on = False
        tests_requested = False
        allure_report = False
        allure_report_ttl = 4
        test_type_filter = None
        junit_report = False
        use_system_python = False
        make_context_on_distbuild = False
        check_dependencies = False
        use_dev_version = False
        create_html_results_resource = True
        ram_drive_size = None
        check_return_code = True
        strip_binaries = sdk2.parameters.Bool('Strip result binaries', default=True)
        use_aapi_fuse = True
        use_arc_instead_of_aapi = False
        aapi_fallback = True
        build_system = consts.SEMI_DISTBUILD_BUILD_SYSTEM
        binary_executor_release_type = "none"
        env_vars = None
        target_platform = None
        ya_add_result = []
        test_filters = None
        report_tests_only = False
        test_log_level = 'debug'
        test_tag = None
        disable_test_timeout = False
        force_build_depends = False
        force_vcs_info_update = False
        ignore_recurses = False
        test_threads = 0
        cache_test_results = False
        tests_retries = 1
        target_platform_flags = None
        host_platform_flags = None
        checkout = False
        sonar = False
        sonar_default_project_filter = False
        sonar_options = None
        sonar_project_filter = None
        failed_tests_cause_error = True
        trace_ya_output = False
        dir_outputs = True
        use_prebuilt_tools = 'default'
        ya_make_extra_parameters = []
        run_tagged_tests_on_yt = False
        yt_store_exclusive = False
        force_sandbox_tags = False
        do_not_remove_resources = sdk2.parameters.Bool("Do not remove resources", required=True, default=False)

    def get_build_type(self):
        return self.Parameters.build_type

    def get_target_type(self, target):
        return common.RTHUB_CM_TARGETS[target]

    def get_target_name(self, target):
        return common.RTHUB_CM_TARGETS[target].name

    def get_target_bin_name(self, target):
        return common.RTHUB_CM_TARGETS[target].bin_name

    def get_target_res_type(self, target):
        return common.RTHUB_CM_TARGETS[target].res_type

    def get_target_arcadia_build_path(self, target):
        return common.RTHUB_CM_TARGETS[target].arcadia_build_path

    def get_targets(self):
        targets = []
        for target in self.Parameters.build_targets:
            if self.get_target_res_type(target) == 'build':
                targets.append(self.get_target_arcadia_build_path(target))
        return targets

    def get_resources(self):
        resources = {}
        for target in self.Parameters.build_targets:
            resource_type = self.get_target_type(target)
            if self.get_target_res_type(target) == 'build':
                resource_path = self.get_target_bin_name(target)
                resources[target] = {
                    'description': target,
                    'resource_type': resource_type,
                    'resource_path': resource_path
                }
            elif self.get_target_res_type(target) == 'vcs':
                resource_path = self.get_target_name(target)
                resources[target] = {
                    'description': target,
                    'resource_type': resource_type,
                    'resource_path': resource_path
                }
        return resources

    def get_arts(self):
        arts = []
        for target in self.Parameters.build_targets:
            if self.get_target_res_type(target) == 'build':
                target_vcs_path = self.get_target_arcadia_build_path(target)
                target_bin_name = self.get_target_bin_name(target)
                arts.append({
                    'path': pj(target_vcs_path, target_bin_name),
                    'dest': ''
                })
        return arts

    def get_arts_source(self):
        arc_dir = sdk2.paths.make_folder('sources')
        src_dir = pj(arc_dir, 'arcadia')
        _, mode = ya_sdk_compat.check_parameters_compatibility(self, self.Parameters.make_context_on_distbuild)
        arc_oauth_token = ya_make_package.prepare_arc_token(self, mode)
        arcadia_ctx, _ = self.get_source_dirs(mode, arc_oauth_token=arc_oauth_token)
        with arcadia_ctx as arcadia_path:
            ya_sdk_compat.ensure_link(arcadia_path, src_dir)

            arts_sources = []
            for target in self.Parameters.build_targets:
                if self.get_target_res_type(target) == 'vcs':
                    target_build_path = self.get_target_arcadia_build_path(target)
                    for entry in target_build_path.split(';'):
                        for item in os.listdir(pj(src_dir, entry)):
                            arts_sources.append({
                                'path': pj(entry, item),
                                'dest': self.get_target_name(target)
                            })
            return arts_sources

    def on_enqueue(self):
        with self.memoize_stage.first_enqueue:
            return YaMake2.on_enqueue(self)

    def on_execute(self):
        with self.memoize_stage.start_child_tasks_and_build:
            for target in self.Parameters.build_targets:
                if self.get_target_res_type(target) == 'package':
                    task = sdk2.Task["YA_PACKAGE"](
                        self,
                        description='Child of {}. Build {}'.format(self.id, self.get_target_name(target)),
                        owner=self.owner,
                        checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                        arcadia_patch=self.Parameters.arcadia_patch,
                        package_type='tarball',
                        packages=self.get_target_arcadia_build_path(target),
                        resource_type=self.get_target_name(target),
                        use_aapi_fuse=True,
                        aapi_fallback=True,
                        raw_package=True,
                        build_type=self.Parameters.build_type,
                        build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                        checkout_mode='auto',
                        sanitize=None if self.Parameters.sanitize_group == 'none' else self.Parameters.sanitize_group,
                        ya_yt_store=True,
                        ya_yt_token_vault_name='robot-zora-yt-token',
                        ya_yt_token_vault_owner='ZORA'
                    )
                    task.Requirements.client_tags = Tag.Group.LINUX
                    task.save().enqueue()
                    self.Context.child_tasks[target] = task.id
            YaMake2.on_execute(self)
        with self.memoize_stage.wait_child_tasks:
            if self.Context.child_tasks:
                raise sdk2.WaitTask(
                    self.Context.child_tasks.values(),
                    ctt.Status.Group.FINISH,
                    wait_all=True
                )

    def on_finish(self, prev_status, status):
        if self.Context.child_tasks:
            for task_key, task_id in self.Context.child_tasks.items():
                task = sdk2.Task[task_id]
                if task.status != ctt.Status.SUCCESS:
                    raise ce.TaskFailure('Child task was finished with status {}'.format(task.status))

            for res_name, build_task_id in self.Context.child_tasks.items():
                resource = sdk2.Resource.find(
                    type=res_name,
                    task_id=build_task_id
                ).first()
                if resource:
                    resource_data = sdk2.ResourceData(resource)
                else:
                    raise ce.TaskFailure('Resource {} with task_id {} not found'.format(res_name, build_task_id))
                resource_new = sdk2.Resource[res_name](
                    self,
                    resource.description,
                    resource_data.path.name,
                    ttl=30
                )
                if not os.path.isdir(str(resource_data.path)):
                    shutil.copy(str(resource_data.path), resource_data.path.name)
                else:
                    shutil.copytree(str(resource_data.path), resource_data.path.name)
                sdk2.ResourceData(resource_new).ready()

    def on_release(self, additional_parameters):
        ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
