import logging
import os
import re
import time

import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2
from sandbox.projects.jupiter.ReleaseExtFiles import JupiterReleaseExtFiles
import fileinput
from os.path import join as pj

class UpdateRTHubResources(sdk2.Task):
    """
    Updates sandbox resources for RTHub.
    Commits new res ids to RTHub packages.
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 60000  # Mb

    class Parameters(sdk2.Task.Parameters):
        resources_list = sdk2.parameters.String(
            "List of resources to update (';'-separated)",
            required=True
        )
        packages_list = sdk2.parameters.String(
            "RTHub packages names in which update resources ids (';'-separated).\nE.g.: \"pages;turbo-pages\".\nDefault: All",
            default="appdocs;images;pages;saas;sitemaps;turbo-pages",
            required=True
        )
        release_machine_component = sdk2.parameters.String(
            "Release machine component name.\nNeeded for merge into branches.",
            default=None
        )
        author = sdk2.parameters.String('Author of commit', default='zomb-sandbox-rw', required=True)
        release_resources = sdk2.parameters.Bool("Release created resources", default=False)
        release_resources_to = sdk2.parameters.String("Release created resources to. ';'-separated", default="")

    def update_resources(self, resources_list):
        logging.info("Launching JupiterReleaseExtFiles sub-task...")
        sub_task = JupiterReleaseExtFiles(
            self,
            description="Updating RTHub resources (by {})".format(self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            bundles_to_build=resources_list,
            create_tar_archive=True,
            force_rebuild=True,
            full_paths_meta=True,
            sandbox_token_name="sb_token",
            sandbox_token_owner="RTHUB",
            set_ttl_inf=False
        ).enqueue()
        self.Context.sub_task_id = sub_task.id
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def update_res_ids(self, file_path, new_res_name, new_res_id):
        for line in fileinput.FileInput(file_path, inplace=True):
            line = re.sub(
                '^FROM_SANDBOX\(FILE [0-9]+ OUT_NOAUTO {}\.tar'.format(new_res_name),
                'FROM_SANDBOX(FILE {} OUT_NOAUTO {}.tar'.format(new_res_id, new_res_name),
                line.rstrip()
            )
            print line

    def release_tasks(self):
        sub_task = self.find(id=self.Context.sub_task_id).first()
        if sub_task.status not in ctt.Status.Group.SUCCEED:
            raise common.errors.TaskFailure("Subtask is failed with status {}".format(sub_task.status))
        else:
            new_res_list = sub_task.Context.downloaded_resources

        if self.Parameters.release_resources:
            rest_client = common.rest.Client()
            if self.Parameters.release_resources_to:
                for d in self.Parameters.release_resources_to.split(';'):
                    logging.info("Releasing {} to {}".format(sub_task.id, d))
                    payload = {
                        "task_id": sub_task.id,
                        "cc": [],
                        "to": [],
                        "params": {},
                        "message": "Auto-release of resources by UpdateRTHubResources",
                        "type": d,
                        "subject": "",
                    }
                    rest_client.release(payload)
                    while rest_client.task[sub_task.id].read()['status'] != 'RELEASED':
                        time.sleep(60)
        return new_res_list

    def commit_new_versions(self, new_res_list, resources_path = None):
        sdk2.svn.Arcadia.checkout(
            url="svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/robot/rthub/packages/resources",
            path=pj(os.curdir, "resources")
        )

        if resources_path is None:
            resources_path = self.Parameters.packages_list

        for res in new_res_list:
            for pkg in resources_path.split(';'):
                file_path = pj(os.curdir, "resources", pkg, "ya.make")
                logging.info("Processing file {}".format(file_path))
                self.update_res_ids(file_path, res, new_res_list[res])

        svn_status = sdk2.svn.Arcadia.status(pj(os.curdir, "resources"))
        logging.info(svn_status)

        ci_msg = "Updated ids of RTHub resources. SKIP_CHECK"
        if self.Parameters.release_machine_component:
            ci_msg += " [mergeto:{}]".format(self.Parameters.release_machine_component)

        try:
            sdk2.svn.Arcadia.commit(
                path=pj(os.curdir, "resources"),
                message=ci_msg,
                user=self.Parameters.author
            )
        except sdk2.vcs.svn.SvnError:
            logging.error("There was a problem with commit!!!")

    def on_execute(self):
        with self.memoize_stage.first_step:
            self.update_resources(self.Parameters.resources_list)
        with self.memoize_stage.second_step:
            new_res_list = self.release_tasks()
            self.commit_new_versions(new_res_list)
