# -*- coding: utf-8 -*-

import logging

from sandbox import common
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.rtmr.common import find_mirrorconfig, LastResource
from sandbox.projects.rtmr.resources import RtmrConfigTool
from sandbox.projects.rtmr.RtmrBuildMirrorsConfig import RtmrBuildMirrorsConfig
from sandbox.sdk2.helpers import subprocess as sp


class RtmrApplyMirrorConfig(sdk2.Task):
    """Apply RTMR mirror config"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 2 * 1024  # 2Gb

    class Parameters(sdk2.Task.Parameters):
        description = "Apply RTMR mirror config"
        kill_timeout = 10 * 60

        config_task = sdk2.parameters.Task(
            "Config task",
            required=True,
            task_type=RtmrBuildMirrorsConfig)

        cluster_name = sdk2.parameters.String("Mirror name", required=True)

        rtmr_config_resource = LastResource(
            "RTMR Config Tool",
            resource_type=RtmrConfigTool,
            required=True
        )

    class Context(sdk2.Task.Context):
        rtmr_config = None

    def update_config_tool(self):
        if self.Context.rtmr_config is not None:
            return
        rtmr_config_resource = self.Parameters.rtmr_config_resource
        self.Context.rtmr_config = str(sdk2.ResourceData(rtmr_config_resource).path)

    def do_rtmr_config(self, args):
        cmd = [self.Context.rtmr_config]
        cmd.extend(args)
        logging.info("Exec rtmr-config with command line %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise Exception("rtmr-config return code: " + str(proc.returncode))

    def on_execute(self):
        self.update_config_tool()

        config_resource = find_mirrorconfig(
            self.Parameters.config_task,
            self.Parameters.cluster_name)

        if config_resource is None:
            raise common.errors.TaskError(
                "Could not find mirror config resource for cluster {cluster_name}".format(
                    cluster_name=self.Parameters.cluster_name))

        mirrorconfig = str(sdk2.ResourceData(config_resource).path)

        self.do_rtmr_config([
            "tables",
            "-c", self.Parameters.cluster_name,
            "-i", mirrorconfig,
            "--yes-please-destroy-production-cluster",
        ])
