# -*- coding: utf-8 -*-

import logging

from sandbox import common
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.common import find_taskconfig, LastResource
from sandbox.projects.rtmr.resources import RtmrConfigTool
from sandbox.sdk2.helpers import subprocess as sp


class RtmrApplyUsertaskConfig(sdk2.Task):
    """Apply usertask config"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 2 * 1024  # 2Gb

    class Parameters(sdk2.Task.Parameters):
        description = "Apply usertask config"
        kill_timeout = 10 * 60

        config_task = sdk2.parameters.Task("Config Task", required=True)
        with sdk2.parameters.String("Cluster name", multiline=True, required=True) as cluster_name:
            _first = True
            for _name in RTMR_CLUSTERS:
                if _first:
                    cluster_name.values[_name] = cluster_name.Value(default=True)
                    _first = False
                else:
                    cluster_name.values[_name] = None

        rtmr_config_resource = LastResource(
            "RTMR Config Tool",
            resource_type=RtmrConfigTool,
            required=True
        )

        with sdk2.parameters.RadioGroup("Filtering options") as filter:
            filter.values.none = filter.Value("none", default=True)
            filter.values.include = filter.Value("include")
            filter.values.exclude = filter.Value("exclude")

        with filter.value["include"]:
            task_ids = sdk2.parameters.String(
                "Comma separated list of task ids to apply config for (empty if "
                "no filtering is needed)")

        with filter.value["exclude"]:
            exclude_tasks = sdk2.parameters.String(
                "Comma separated list of task ids to exclude from applying config "
                "(empty if no filtering is needed)")

    class Context(sdk2.Task.Context):
        rtmr_config = None

    def update_config_tool(self):
        if self.Context.rtmr_config is not None:
            return
        rtmr_config_resource = self.Parameters.rtmr_config_resource
        self.Context.rtmr_config = str(sdk2.ResourceData(rtmr_config_resource).path)

    def do_rtmr_config(self, args):
        cmd = [self.Context.rtmr_config]
        cmd.extend(args)
        logging.info("Exec rtmr-config with command line %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise Exception("rtmr-config return code: " + str(proc.returncode))

    def on_execute(self):
        self.update_config_tool()
        config_resource = find_taskconfig(self.Parameters.config_task, self.Parameters.cluster_name)
        if config_resource is None:
            raise common.errors.TaskError(
                "Could not find task config resource")

        taskconfig = str(sdk2.ResourceData(config_resource).path)

        args = [
            "tasks",
            "-c", self.Parameters.cluster_name,
            "-i", taskconfig,
            "--yes-please-destroy-production-cluster",
        ]

        if self.Parameters.task_ids:
            task_ids = filter(bool, [task_id.strip() for task_id in self.Parameters.task_ids.split(",")])
            if task_ids:
                for task_id in task_ids:
                    args.extend(["--update-tasks", task_id])
        elif self.Parameters.exclude_tasks:
            exclude_tasks = filter(bool, [task_id.strip() for task_id in self.Parameters.exclude_tasks.split(",")])
            if exclude_tasks:
                for task_id in exclude_tasks:
                    args.extend(["--exclude-tasks", task_id])

        self.do_rtmr_config(args)
