# -*- coding: utf-8 -*-

import json
import logging
import os
import re

import requests

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.client as ctc
from sandbox.projects.geobase.Geodata5BinStable import resource as gbr
from sandbox.projects.rtmr.common import wait_tasks, get_arcadia_url, get_version, LastResource, arcadia_checkout
from sandbox.projects.rtmr.resources import RtmrUserdata, RtmrUserdataDeb
from sandbox.sandboxsdk.copy import RemoteCopyRsync


class RtmrBuildCommonUserdata(sdk2.Task):
    """Build RTMR Common Userdata"""

    PACKAGE_PATH = "rtmapreduce/packages/yandex-search-rtmr-common-userdata"
    PUBLISH_REPOSITORY = "search"

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.IPV6
        disk_space = 20 * 1024  # 20Gb

    class Parameters(sdk2.Task.Parameters):
        description = "Build RTMR Common Userdata"
        kill_timeout = 5 * 3600

        geodata = LastResource(
            "Geodata5 resource",
            resource_type=gbr.GEODATA5BIN_STABLE,
            required=False
        )
        statface_username = sdk2.parameters.String(
            "Statface username",
            required=True,
            default_value="robot_robot-rtmr-build"
        )
        statface_password_vault_owner = sdk2.parameters.String(
            "Vault secret owner with Statface password",
            required=True,
            default_value="RTMR-DEV"
        )
        statface_password_vault = sdk2.parameters.String(
            "Vault secret name with Statface password",
            required=True,
            default_value="robot-rtmr-build-statface"
        )
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia url",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )
        svn_username = sdk2.parameters.String(
            "Author of commit",
            required=True,
            default_value="zomb-sandbox-rw"
        )
        ya_package_name_for_sign = sdk2.parameters.String(
            "Name of the user to sign the package",
            required=False,
        )
        build_disk_space = sdk2.parameters.Integer("Build execution space (Gb)", default_value=60)

    class Context(sdk2.Task.Context):
        resources = list()
        build_tasks = list()
        arcadia_branch = None
        arcadia_revision = None

    def get_geodata(self):
        resource = self.Parameters.geodata
        if resource is None or resource == "0":
            resource = sdk2.Resource.find(resource_type=gbr.GEODATA5BIN_STABLE).order(-sdk2.Resource.id).first()
        return resource.id

    def make_blockstat_resource(self):
        filename = "blockstat.dict"
        resource = RtmrUserdata(
            self,
            "RTMR Common Userdata " + filename,
            filename
        )
        resource_data = sdk2.ResourceData(resource)
        headers = dict()
        headers["StatRobotUser"] = self.Parameters.statface_username
        headers["StatRobotPassword"] = sdk2.Vault.data(
            self.Parameters.statface_password_vault_owner,
            self.Parameters.statface_password_vault,
        )
        request = requests.get(
            "https://upload.stat.yandex-team.ru/_api/dictionary?name=blockstat&format=legacy",
            stream=True,
            headers=headers
        )

        dst = str(resource_data.path.absolute())
        with open(dst, "w+") as fd:
            for chunk in request.iter_content(chunk_size=50 * 1024):
                if chunk:
                    fd.write(chunk)
        resource_data.ready()
        return resource.id

    def update_package(self, resources):
        svn_dir_obj = sdk2.Path("package-src")
        svn_dir_obj.mkdir()
        svn_dir = str(svn_dir_obj.absolute())

        parsed_url = sdk2.svn.Arcadia.parse_url(self.Parameters.arcadia_url)
        if parsed_url.branch is None:
            self.Context.arcadia_branch = "trunk"
        else:
            self.Context.arcadia_branch = parsed_url.branch

        arcadia_checkout(
            sdk2.svn.Arcadia.replace(
                self.Parameters.arcadia_url,
                path=os.path.join(parsed_url.path, self.PACKAGE_PATH),
            ),
            svn_dir,
            sdk2.svn.Arcadia.Depth.IMMEDIATES,
        )
        template_path = os.path.join(svn_dir, "base-pkg.json")
        out_path = os.path.join(svn_dir, "pkg.json")
        try:
            with open(template_path, "r") as fd:
                template = json.load(fd)
        except IOError:
            raise common.errors.TaskError("Error read package template file")
        except ValueError:
            raise common.errors.TaskError("Package template contains invalid json")

        pkg_data = template.get("data")
        if pkg_data is None:
            raise common.errors.TaskError("Invalid package template not contain \"data\" section")
        for resource in resources:
            resource_info = dict(
                destination=dict(path="/place/berkanavt/rtmr/"),
                source=dict(type="SANDBOX_RESOURCE", id=resource),
            )
            pkg_data.append(resource_info)

        try:
            with open(out_path, "w+") as fd:
                json.dump(template, fd, indent=4)
        except IOError:
            raise common.errors.TaskError("Error write package file")

        logging.info("Arcadia status...\n%s", sdk2.svn.Arcadia.status(svn_dir))
        logging.info("Arcadia diff...\n%s", sdk2.svn.Arcadia.diff(svn_dir))

        comment = "{task_type} update package resources\nhttps://sandbox.yandex-team.ru/task/{task_id}/view".format(
            task_type=self.type,
            task_id=self.id
        )
        commit_output = sdk2.svn.Arcadia.commit(svn_dir, comment, user=self.Parameters.svn_username)
        found_items = re.search(r'^Committed revision (?P<revision>\d+)\.$', commit_output, re.M)
        if found_items is None:
            raise common.errors.TaskError("Commit revison not found")
        commited_revision = found_items.group(1)
        self.Context.arcadia_revision = commited_revision

    def make_resource(self, src, name):
        resource = RtmrUserdata(self, "RTMR Common Userdata " + name, name)
        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())
        RemoteCopyRsync(src, dst)()
        resource_data.ready()
        return resource.id

    def build_package(self):
        if self.Context.arcadia_revision is None:
            raise common.errors.TaskError("Revision not found")
        build_task = sdk2.Task["YA_PACKAGE"](
            self,
            description="Build RTMR Common Userdata " + get_version(self),
            priority=self.Parameters.priority,
            checkout_arcadia_from_url=get_arcadia_url(self, self.Parameters.arcadia_url),
            use_new_format=True,
            packages=os.path.join(self.PACKAGE_PATH, "pkg.json"),
            host_platform="linux",
            package_type="debian",
            publish_package=True,
            publish_to=self.PUBLISH_REPOSITORY,
            checkout_mode="auto",
            resource_type=RtmrUserdataDeb.name,
            key_user=self.Parameters.ya_package_name_for_sign,
        )
        build_task.Requirements.disk_space = self.Parameters.build_disk_space * 1024
        build_task.save().enqueue()
        self.Context.build_tasks.append(build_task.id)
        self.Context.save()

    def on_execute(self):
        with self.memoize_stage.build_resources(commit_on_entrance=False):
            resources = list()
            resources.append(self.get_geodata())
            resources.append(self.make_blockstat_resource())
            resources.append(self.make_resource("rsync://veles02.search.yandex.net/Berkanavt/geodata/geohelper", "geohelper"))
            resources.append(self.make_resource("rsync://veles02.search.yandex.net/Berkanavt/geodata/user_type_resolver",
                                                "user_type_resolver"))
            self.Context.resources.extend(resources)
            self.Context.save()

        with self.memoize_stage.update_package(commit_on_entrance=False):
            self.update_package(self.Context.resources)

        with self.memoize_stage.build_package(commit_on_entrance=False):
            self.build_package()
        wait_tasks(self.Context.build_tasks)
