# -*- coding: utf-8 -*-

import json
import logging
import os

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.RtmrBuildPackage import RtmrBuildPackage
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.resources import RtmrUsertaskDeb, RtmrPackageConfig, RtmrUsertaskConfig, RtmrGraphConfig
from sandbox.sdk2.helpers import subprocess as sp


class RtmrBuildGraph(sdk2.Task):
    """Build RTMR Graph"""

    RTMR_PACKAGES_PATH = "rtmapreduce/packages/tasks"

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 10 * 1024  # 10Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Build RTMR Graph"
        kill_timeout = 5 * 3600

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )
        parallel_limit = sdk2.parameters.Integer(
            "Parallel build package limit",
            required=True,
            default_value=99
        )
        graphs = sdk2.parameters.String("Names of graphs for tasks that need to be built (graph1,graph2...)", required=True)
        build_packages = sdk2.parameters.Bool("Build packages", default=True)
        with sdk2.parameters.CheckGroup("Clusters") as clusters:
            for _name in RTMR_CLUSTERS:
                clusters.values[_name] = _name
        build_disk_space = sdk2.parameters.Integer("Build execution space (Gb)", default_value=60)
        ya_package_name_for_sign = sdk2.parameters.String("Name of the user to sign the package", required=False)
        strip_packages = sdk2.parameters.Bool("Strip debug info", default_value=True)
        create_debug_packages = sdk2.parameters.Bool("Create debug info packages", default_value=False)
        build_type = sdk2.parameters.String("Build type", default_value="release", choices=[("release", "release"), ("profile", "profile")])

    class Context(sdk2.Task.Context):
        build_task = None
        rtmr_gencfg_path = None
        rtmr_configs_path = None
        packages = None
        arcadia_branch = None
        arcadia_revision = None

    def get_clusters(self):
        clusters = set(self.Parameters.clusters)
        return list(clusters)

    def get_cluster_config_path(self, cluster_name):
        return os.path.join(
            rtmr_common.get_rtmr_configs(self, rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url)),
            cluster_name + ".cfg"
        )

    def get_packages_for_cluster(self, cluster_name):
        cmd = [
            self.Context.rtmr_gencfg_path,
            "-p", "PackagesByGraph",
            "-c", self.get_cluster_config_path(cluster_name),
            "--graphs", self.Parameters.graphs,
        ]

        logging.info("Generate packages list for cluster %s", cluster_name)
        logging.info("run gencfg %r", cmd)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-gencfg")) as pl:
            proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=pl.stdout)
            stdout, _ = proc.communicate()
        logging.info("gencfg output %r", stdout)
        if proc.returncode != 0:
            raise common.errors.TaskError("Gencfg return code: " + str(proc.returncode))
        try:
            return json.loads(stdout)
        except ValueError:
            raise common.errors.TaskError("Gencfg return invalid json")

    def save_tasks_config(self, cluster_name):
        resource = RtmrUsertaskConfig(
            self,
            "RTMR tasks config for cluster {cluster} {version}".format(
                cluster=cluster_name,
                version=rtmr_common.get_version(self)
            ),
            "taskconfig-{}.json".format(cluster_name),
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
            cluster=cluster_name,
        )
        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())
        cmd = [
            self.Context.rtmr_gencfg_path,
            "-p", "TaskDataByGraph",
            "-c", self.get_cluster_config_path(cluster_name),
            "--graphs", self.Parameters.graphs,
        ]

        logging.info("Generate taskconfig for cluster %s", cluster_name)
        logging.info("run gecfg %r", cmd)
        with open(dst, "w+") as fd:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-gencfg")) as pl:
                proc = sp.Popen(cmd, stdout=fd, stderr=pl.stdout)
                proc.wait()
        if proc.returncode != 0:
            raise common.errors.TaskError("Gencfg return code: " + str(proc.returncode))
        resource_data.ready()

    def save_graphs_config(self, cluster_name):
        resource = RtmrGraphConfig(
            self,
            "RTMR graphs config for cluster {cluster} {version}".format(
                cluster=cluster_name,
                version=rtmr_common.get_version(self)
            ),
            "graphsconfig-{}.json".format(cluster_name),
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
            cluster=cluster_name,
        )
        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())
        cmd = [
            self.Context.rtmr_gencfg_path,
            "-p", "Graph",
            "-c", self.get_cluster_config_path(cluster_name),
            "--graphs", self.Parameters.graphs,
        ]

        logging.info("Generate graphsconfig for cluster %s", cluster_name)
        logging.info("run gecfg %r", cmd)
        with open(dst, "w+") as fd:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-gencfg")) as pl:
                proc = sp.Popen(cmd, stdout=fd, stderr=pl.stdout)
                proc.wait()
        if proc.returncode != 0:
            raise common.errors.TaskError("Gencfg return code: " + str(proc.returncode))
        resource_data.ready()

    def save_packages_config(self, packages):
        resource = RtmrPackageConfig(
            self,
            "RTMR packages config " + rtmr_common.get_version(self),
            "packages-config.json",
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
        )
        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())
        with open(dst, "w+") as fd:
            json.dump(packages, fd, indent=2)
        resource_data.ready()

    def make_package_configs(self):
        all_packages = set()
        packages = dict()
        for cluster_name in self.get_clusters():
            self.save_tasks_config(cluster_name)
            self.save_graphs_config(cluster_name)
            cluster_packages = self.get_packages_for_cluster(cluster_name)
            packages[cluster_name] = cluster_packages
            all_packages.update(set(cluster_packages))
        self.set_info("Packages:\n " + "\n ".join(all_packages))
        self.Context.packages = list(all_packages)
        self.save_packages_config(packages)

    def build_packages(self):
        packages_paths = map(lambda p: os.path.join(self.RTMR_PACKAGES_PATH, p, "pkg.json"), self.Context.packages)
        task = RtmrBuildPackage(
            self,
            description="Build RTMR Graph {version}\n{packages}".format(
                version=rtmr_common.get_version(self),
                packages="\n".join(self.Context.packages),
            ),
            priority=self.Parameters.priority,
            arcadia_url=rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url),
            packages=packages_paths,
            resource_type=RtmrUsertaskDeb.name,
            parallel_limit=self.Parameters.parallel_limit,
            build_disk_space=self.Parameters.build_disk_space,
            strip_packages=self.Parameters.strip_packages,
            create_debug_packages=self.Parameters.create_debug_packages,
            build_type=self.Parameters.build_type,
            publish_package=False,
            package_type="debian",
            ya_package_name_for_sign=self.Parameters.ya_package_name_for_sign,
        )
        task.save().enqueue()
        self.Context.build_task = task.id
        self.Context.save()

    def on_execute(self):
        with self.memoize_stage.update_version(commit_on_entrance=False):
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        with self.memoize_stage.build_gencfg(commit_on_entrance=False):
            rtmr_common.get_rtmr_gencfg(self, rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url))

        with self.memoize_stage.getting_packages(commit_on_entrance=False):
            self.make_package_configs()

        with self.memoize_stage.build_packages(commit_on_entrance=False):
            if self.Parameters.build_packages and self.Context.packages:
                self.build_packages()

        if self.Parameters.build_packages and self.Context.packages:
            rtmr_common.wait_tasks([self.Context.build_task])
