# -*- coding: utf-8 -*-

import json
import logging
import os

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.resources import RtmrPackageConfig, RtmrUsertaskConfig
from sandbox.sdk2.helpers import subprocess as sp


class RtmrBuildGraphConfig(sdk2.Task):
    """Build RTMR Graph config"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Build RTMR Graph config"
        kill_timeout = 2 * 3600

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )
        graphs = sdk2.parameters.String("Names of graphs for tasks that need to be built (graph1,graph2...)")
        with sdk2.parameters.CheckGroup("Clusters") as clusters:
            for _name in RTMR_CLUSTERS:
                clusters.values[_name] = _name

    class Context(sdk2.Task.Context):
        rtmr_gencfg_path = None
        rtmr_configs_path = None
        packages = None
        arcadia_branch = None
        arcadia_revision = None

    def get_clusters(self):
        clusters = set(self.Parameters.clusters)
        return list(clusters)

    def get_cluster_config_path(self, cluster_name):
        return os.path.join(
            rtmr_common.get_rtmr_configs(self, rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url)),
            cluster_name + ".cfg"
        )

    def get_packages_for_cluster(self, cluster_name):
        cmd = [
            self.Context.rtmr_gencfg_path,
            "-p", "PackagesByGraph",
            "-c", self.get_cluster_config_path(cluster_name),
            "--graphs", self.Parameters.graphs,
        ]

        logging.info("Generate packages list for cluster %s", cluster_name)
        logging.info("run gencfg %r", cmd)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-gencfg")) as pl:
            proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=pl.stdout)
            stdout, _ = proc.communicate()
        logging.info("gencfg output %r", stdout)
        if proc.returncode != 0:
            raise common.errors.TaskError("Gencfg return code: " + str(proc.returncode))
        try:
            return json.loads(stdout)
        except ValueError:
            raise common.errors.TaskError("Gencfg return invalid json")

    def save_tasks_config(self, cluster_name):
        resource = RtmrUsertaskConfig(
            self,
            "RTMR task config for cluster {cluster} {version}".format(
                cluster=cluster_name,
                version=rtmr_common.get_version(self)
            ),
            "taskconfig-{}.json".format(cluster_name),
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
            cluster=cluster_name,
        )
        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())
        cmd = [
            self.Context.rtmr_gencfg_path,
            "-p", "TaskDataByGraph",
            "-c", self.get_cluster_config_path(cluster_name),
            "--graphs", self.Parameters.graphs,
        ]

        logging.info("Generate taskconfig for cluster %s", cluster_name)
        logging.info("run gecfg %r", cmd)
        with open(dst, "w+") as fd:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-gencfg")) as pl:
                proc = sp.Popen(cmd, stdout=fd, stderr=pl.stdout)
                proc.wait()
        if proc.returncode != 0:
            raise common.errors.TaskError("Gencfg return code: " + str(proc.returncode))
        resource_data.ready()

    def save_packages_config(self, packages):
        resource = RtmrPackageConfig(
            self,
            "RTMR packages config " + rtmr_common.get_version(self),
            "packages-config.json",
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
        )
        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())
        with open(dst, "w+") as fd:
            json.dump(packages, fd, indent=2)
        resource_data.ready()

    def make_package_configs(self):
        all_packages = set()
        packages = dict()
        for cluster_name in self.get_clusters():
            self.save_tasks_config(cluster_name)
            cluster_packages = self.get_packages_for_cluster(cluster_name)
            packages[cluster_name] = cluster_packages
            all_packages.update(set(cluster_packages))
        self.set_info("Packages:\n " + "\n ".join(all_packages))
        self.Context.packages = list(all_packages)
        self.save_packages_config(packages)

    def on_execute(self):
        with self.memoize_stage.update_version(commit_on_entrance=False):
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        with self.memoize_stage.build_gencfg(commit_on_entrance=False):
            rtmr_common.get_rtmr_gencfg(self, rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url))

        with self.memoize_stage.getting_packages(commit_on_entrance=False):
            self.make_package_configs()
