# -*- coding: utf-8 -*-

import logging
import os

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox.projects.rtmr.clusters import RtmrClustersInfo
from sandbox.projects.rtmr.resources import RtmrMirrorConfig
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class RtmrBuildMirrorsConfig(sdk2.Task):
    """Build RTMR mirrors config"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Build RTMR mirrors config"

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )

        mirrors = sdk2.parameters.List(
            "List of mirrors to generate config for (all mirrors if not set)")

        update_collocations = sdk2.parameters.Bool(
            "Update collocations",
            default=False)

    class Context(sdk2.Task.Context):
        rtmr_gencfg_path = None
        rtmr_configs_path = None
        arcadia_branch = None
        arcadia_revision = None

    def get_mirrors(self):
        if self.Parameters.mirrors:
            return self.Parameters.mirrors

        mirrors = []
        for cluster_name, cluster_info in RtmrClustersInfo().clusters.items():
            if cluster_info is None:
                continue

            if cluster_info.is_mirror:
                mirrors.append(cluster_name)

        return mirrors

    def get_cluster_config_path(self, cluster_name):
        return os.path.join(
            rtmr_common.get_rtmr_configs(self, rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url)),
            cluster_name + ".cfg"
        )

    def save_tables_config(self, cluster_name):
        resource = RtmrMirrorConfig(
            self,
            "RTMR mirror config for cluster {cluster} {version}".format(
                cluster=cluster_name,
                version=rtmr_common.get_version(self)
            ),
            "tableconfig-{}.json".format(cluster_name),
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
            cluster=cluster_name,
        )

        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())

        cmd = [
            self.Context.rtmr_gencfg_path,
            "-p", "Tables",
            "-c", self.get_cluster_config_path(cluster_name),
        ]

        if self.Parameters.update_collocations:
            cmd.append("--update-collocations")

        logging.info("Generate table config for cluster %s", cluster_name)
        logging.info("run gecfg %r", cmd)

        with open(dst, "w+") as fd:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-gencfg")) as pl:
                proc = sp.Popen(cmd, stdout=fd, stderr=pl.stdout)
                proc.wait()

        if proc.returncode != 0:
            raise common.errors.TaskError("Gencfg return code: " + str(proc.returncode))

        resource_data.ready()

    def make_table_configs(self):
        for mirror in self.get_mirrors():
            self.save_tables_config(mirror)

    def on_execute(self):
        with self.memoize_stage.update_version(commit_on_entrance=False):
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        with self.memoize_stage.build_gencfg(commit_on_entrance=False):
            rtmr_common.get_rtmr_gencfg(
                self,
                rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url))

        with self.memoize_stage.make_table_configs(commit_on_entrance=False):
            self.make_table_configs()
