# -*- coding: utf-8 -*-

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.resources import RtmrReleaseDeb


class RtmrBuildPackage(sdk2.Task):
    """Build RTMR Package"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1 * 1024  # 1Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Build RTMR Package"
        kill_timeout = 5 * 3600

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )
        packages = sdk2.parameters.List("Package paths", sdk2.parameters.String)
        ya_package_name_for_sign = sdk2.parameters.String("Name of the user to sign the package", required=False)
        parallel_limit = sdk2.parameters.Integer(
            "Parallel build package limit",
            required=True,
            default_value=1
        )
        resource_type = sdk2.parameters.String(
            "Package resource type",
            required=True,
            default_value=RtmrReleaseDeb.name
        )
        build_disk_space = sdk2.parameters.Integer("Build execution space (Gb)", default_value=60)

        with sdk2.parameters.String("Package type: debian or tarball", multiline=True, required=True) as package_type:
            package_type.values["debian"] = package_type.Value(default=True)
            package_type.values["tarball"] = None

        clear_build = sdk2.parameters.Bool("Clear build", required=True, default_value=False)
        with clear_build.value[True]:
            semi_clear_build = sdk2.parameters.Bool("Clear build cache directory only", default_value=False)

        strip_packages = sdk2.parameters.Bool("Strip debug info", default_value=False)
        create_debug_packages = sdk2.parameters.Bool("Create debug info packages", default_value=False)
        build_type = sdk2.parameters.String("Build type", default_value="release", choices=[("release", "release"), ("profile", "profile")])

        with sdk2.parameters.Group("Publishing") as publishing_block:
            publish_package = sdk2.parameters.Bool("Publish package", default_value=True, required=True)

        with sdk2.parameters.Group("Tests") as tests_block:
            run_tests = sdk2.parameters.Bool("Run tests after build", required=True, default_value=False)
            with run_tests.value[True]:
                run_long_tests = sdk2.parameters.Bool("Run long tests", default_value=False)
                ignore_fail_tests = sdk2.parameters.Bool("Ignore fail tests", default_value=False)

        with sdk2.parameters.Group("YT store params") as yt_block:
            ya_yt_store = sdk2.parameters.Bool("Use YT store", required=True, default_value=True)
            with ya_yt_store.value[True]:
                ya_yt_proxy = sdk2.parameters.String("YT store proxy", default_value="freud.yt.yandex.net")
                ya_yt_dir = sdk2.parameters.String(
                    "YT store cypress path",
                    default_value="//home/rtmr-dev/build-cache"
                )
                ya_yt_token_vault_owner = sdk2.parameters.String(
                    "YT token vault owner",
                    default_value="RTMR-DEV"
                )
                ya_yt_token_vault_name = sdk2.parameters.String(
                    "YT token vault name",
                    default_value="robot-rtmr-build-yt"
                )
                ya_yt_put = sdk2.parameters.Bool("Upload build artifacts to YT store", default_value=True)
                with ya_yt_put.value[True]:
                    ya_yt_store_codec = sdk2.parameters.String("YT store codec", default_value="zstd08_1")
                    ya_yt_replace_result = sdk2.parameters.Bool(
                        "Build only targets that need to be uploaded to the YT store",
                        default_value=True
                    )
                    ya_yt_max_cache_size = sdk2.parameters.Integer(
                        "YT max cache size (bytes)",
                        default_value=68003648853  # 63Gb
                    )

    class Context(sdk2.Task.Context):
        build_tasks = list()
        arcadia_branch = None
        arcadia_revision = None

    def build_packages(self, packages):
        parameters = dict(
            kill_timeout=self.Parameters.kill_timeout - 5 * 60,
            ya_timeout=self.Parameters.kill_timeout - 5 * 60,
            checkout_arcadia_from_url=rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url),
            use_new_format=True,
            packages=";\n".join(packages),
            host_platform="linux",
            package_type=self.Parameters.package_type,
            publish_package=self.Parameters.publish_package,
            checkout_mode="auto",
            resource_type=self.Parameters.resource_type,
            run_tests=self.Parameters.run_tests,
            ya_yt_store=self.Parameters.ya_yt_store,
            clear_build=self.Parameters.clear_build,
            strip_binaries=self.Parameters.strip_packages,
            create_debug_packages=self.Parameters.create_debug_packages,
            build_type=self.Parameters.build_type,
            key_user=self.Parameters.ya_package_name_for_sign,
            package_resource_attrs={
                'ttl': 14
            }
        )
        if self.Parameters.publish_package:
            parameters["publish_to"] = "search"

        if self.Parameters.run_tests:
            parameters["run_long_tests"] = self.Parameters.run_long_tests
            parameters["ignore_fail_tests"] = self.Parameters.ignore_fail_tests

        if self.Parameters.ya_yt_store:
            parameters["ya_yt_proxy"] = self.Parameters.ya_yt_proxy
            parameters["ya_yt_dir"] = self.Parameters.ya_yt_dir
            parameters["ya_yt_token_vault_name"] = self.Parameters.ya_yt_token_vault_name
            parameters["ya_yt_token_vault_owner"] = self.Parameters.ya_yt_token_vault_owner
            parameters["ya_yt_put"] = self.Parameters.ya_yt_put
            if self.Parameters.ya_yt_put:
                parameters["ya_yt_store_codec"] = self.Parameters.ya_yt_store_codec
                parameters["ya_yt_replace_result"] = self.Parameters.ya_yt_replace_result
                parameters["ya_yt_max_cache_size"] = self.Parameters.ya_yt_max_cache_size

        if self.Parameters.clear_build:
            parameters["semi_clear_build"] = self.Parameters.semi_clear_build

        build_task = sdk2.Task["YA_PACKAGE"](
            self,
            owner=self.owner,
            priority=self.Parameters.priority,
            description="Build RTMR Package {version}\n{packages}".format(
                version=rtmr_common.get_version(self),
                packages="\n".join(packages),
            ),
            **parameters
        )
        build_task.Requirements.disk_space = self.Parameters.build_disk_space * 1024
        build_task.save().enqueue()
        self.Context.build_tasks.append(build_task.id)
        self.Context.save()

    def collect_packages(self):
        packages = list()
        for task in self.find():
            for resource in sdk2.Resource.find(task=task).limit(0):
                if resource.type != self.Parameters.resource_type:
                    continue
                if hasattr(resource, "resource_version"):
                    packages.append((resource.resource_name, resource.resource_version))
        return packages

    def packages_info(self, resources):
        return "\n".join([name + "=" + version for (name, version) in resources])

    def show_packages(self):
        self.set_info("Packages:\n" + self.packages_info(self.collect_packages()))

    def on_execute(self):
        if self.Parameters.parallel_limit < 1:
            raise common.errors.TaskError("Invalid parallel limit, it must be greater that 0")
        if len(self.Parameters.packages) == 0:
            raise common.errors.TaskError("Empty package list")

        with self.memoize_stage.update_version(commit_on_entrance=False):
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        with self.memoize_stage.build_packages(commit_on_entrance=False):
            packages = self.Parameters.packages
            while len(packages) > 0:
                to_build = packages[:self.Parameters.parallel_limit]
                packages = packages[self.Parameters.parallel_limit:]
                self.build_packages(to_build)

        with self.memoize_stage.wait_build(commit_on_entrance=False, commit_on_wait=False):
            rtmr_common.wait_tasks(self.Context.build_tasks)
            self.Context.build_tasks = list()
            self.Context.save()

        self.show_packages()
