# -*- coding: utf-8 -*-

import logging
import os

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.resources import RtmrSourceConfig
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class RtmrBuildSourcesConfig(sdk2.Task):
    """Build RTMR sources config"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Build RTMR sources config"

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )

        with sdk2.parameters.CheckGroup("Clusters") as clusters:
            for _name in RTMR_CLUSTERS:
                clusters.values[_name] = _name

    class Context(sdk2.Task.Context):
        rtmr_gencfg_path = None
        rtmr_configs_path = None
        arcadia_branch = None
        arcadia_revision = None

    def get_clusters(self):
        clusters = set(self.Parameters.clusters)
        return list(clusters)

    def get_cluster_config_path(self, cluster_name):
        return os.path.join(
            rtmr_common.get_rtmr_configs(self, rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url)),
            cluster_name + ".cfg"
        )

    def save_sources_config(self, cluster_name):
        resource = RtmrSourceConfig(
            self,
            "RTMR source config for cluster {cluster} {version}".format(
                cluster=cluster_name,
                version=rtmr_common.get_version(self)
            ),
            "sourceconfig-{}.json".format(cluster_name),
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
            cluster=cluster_name,
        )

        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())

        cmd = [
            self.Context.rtmr_gencfg_path,
            "-p", "Sources",
            "-c", self.get_cluster_config_path(cluster_name),
        ]

        logging.info("Generate source config for cluster %s", cluster_name)
        logging.info("run gecfg %r", cmd)

        with open(dst, "w+") as fd:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-gencfg")) as pl:
                proc = sp.Popen(cmd, stdout=fd, stderr=pl.stdout)
                proc.wait()

        if proc.returncode != 0:
            raise common.errors.TaskError("Gencfg return code: " + str(proc.returncode))

        resource_data.ready()

    def make_source_configs(self):
        for cluster_name in self.get_clusters():
            self.save_sources_config(cluster_name)

    def on_execute(self):
        with self.memoize_stage.update_version(commit_on_entrance=False):
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        with self.memoize_stage.build_gencfg(commit_on_entrance=False):
            rtmr_common.get_rtmr_gencfg(
                self,
                rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url))

        with self.memoize_stage.make_source_configs(commit_on_entrance=False):
            self.make_source_configs()
