# -*- coding: utf-8 -*-

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import sdk2
from sandbox.projects.rtmr.resources import RtmrPqoutPackage, RtmrYdboutPackage, \
    RtmrSolomonoutPackage


class BuildTarget:
    def __init__(self, name, resource_type):
        self.name = name
        self.resource_type = resource_type


class RtmrBuildSystemTasks(sdk2.Task):
    """Build RTMR system tasks"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Build RTMR system tasks"
        kill_timeout = 3 * 3600  # 3 hours

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )

        with sdk2.parameters.Group("Targets") as targets_block:
            build_pqout = sdk2.parameters.Bool(
                "Build pqout",
                required=False,
                default_value=True)

            build_ydbout = sdk2.parameters.Bool(
                "Build ydbout",
                required=False,
                default_value=True)

            build_solomonout = sdk2.parameters.Bool(
                "Build solomonout",
                required=False,
                default_value=True)

        with sdk2.parameters.Group("Build settings") as build_block:
            ya_package_name_for_sign = sdk2.parameters.String(
                "Name of the user to sign the package",
                required=False)

            parallel_limit = sdk2.parameters.Integer(
                "Parallel build package limit",
                required=True,
                default_value=5
            )

            build_disk_space = sdk2.parameters.Integer(
                "Build execution space (Gb)",
                default_value=60)

            clear_build = sdk2.parameters.Bool(
                "Clear build",
                required=True,
                default_value=False)

            with clear_build.value[True]:
                semi_clear_build = sdk2.parameters.Bool(
                    "Clear build cache directory only",
                    default_value=False)

            strip_packages = sdk2.parameters.Bool(
                "Strip debug info",
                default_value=False)

            create_debug_packages = sdk2.parameters.Bool(
                "Create debug info packages",
                default_value=False)

        with sdk2.parameters.String("Package type: debian or tarball", multiline=True, required=True) as package_type:
            package_type.values["tarball"] = package_type.Value(default=True)
            package_type.values["debian"] = None

        with sdk2.parameters.Group("Tests") as tests_block:
            run_tests = sdk2.parameters.Bool("Run tests after build", required=True, default_value=False)
            with run_tests.value[True]:
                run_long_tests = sdk2.parameters.Bool("Run long tests", default_value=False)
                ignore_fail_tests = sdk2.parameters.Bool("Ignore fail tests", default_value=False)

        with sdk2.parameters.Group("YT store params") as yt_block:
            ya_yt_store = sdk2.parameters.Bool("Use YT store", required=True, default_value=True)
            with ya_yt_store.value[True]:
                ya_yt_proxy = sdk2.parameters.String("YT store proxy", default_value="freud.yt.yandex.net")
                ya_yt_dir = sdk2.parameters.String(
                    "YT store cypress path",
                    default_value="//home/rtmr-dev/build-cache"
                )
                ya_yt_token_vault_owner = sdk2.parameters.String(
                    "YT token vault owner",
                    default_value="RTMR-DEV"
                )
                ya_yt_token_vault_name = sdk2.parameters.String(
                    "YT token vault name",
                    default_value="robot-rtmr-build-yt"
                )
                ya_yt_put = sdk2.parameters.Bool("Upload build artifacts to YT store", default_value=True)
                with ya_yt_put.value[True]:
                    ya_yt_store_codec = sdk2.parameters.String("YT store codec", default_value="zstd08_1")
                    ya_yt_replace_result = sdk2.parameters.Bool(
                        "Build only targets that need to be uploaded to the YT store",
                        default_value=True
                    )
                    ya_yt_max_cache_size = sdk2.parameters.Integer(
                        "YT max cache size (bytes)",
                        default_value=68003648853  # 63Gb
                    )

    class Context(sdk2.Task.Context):
        build_tasks = list()
        arcadia_branch = None
        arcadia_revision = None

    def build_target(self, target):
        parameters = dict(
            kill_timeout=self.Parameters.kill_timeout - 5 * 60,
            ya_timeout=self.Parameters.kill_timeout - 5 * 60,
            checkout_arcadia_from_url=rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url),
            packages="rtmapreduce/packages/tasks/yandex-search-rtmr-task-" + target.name + "/pkg.json",
            use_new_format=True,
            package_type=self.Parameters.package_type,
            publish_package=False,
            host_platform="linux",
            target_platform="linux",
            checkout_mode="auto",
            resource_type=target.resource_type.name,
            do_not_remove_resources=True,
            run_tests=self.Parameters.run_tests,
            ya_yt_store=self.Parameters.ya_yt_store,
            clear_build=self.Parameters.clear_build,
            strip_binaries=self.Parameters.strip_packages,
            create_debug_packages=self.Parameters.create_debug_packages,
            key_user=self.Parameters.ya_package_name_for_sign,
        )

        if self.Parameters.run_tests:
            parameters["run_long_tests"] = self.Parameters.run_long_tests
            parameters["ignore_fail_tests"] = self.Parameters.ignore_fail_tests

        if self.Parameters.ya_yt_store:
            parameters["ya_yt_proxy"] = self.Parameters.ya_yt_proxy
            parameters["ya_yt_dir"] = self.Parameters.ya_yt_dir
            parameters["ya_yt_token_vault_name"] = self.Parameters.ya_yt_token_vault_name
            parameters["ya_yt_token_vault_owner"] = self.Parameters.ya_yt_token_vault_owner
            parameters["ya_yt_put"] = self.Parameters.ya_yt_put
            if self.Parameters.ya_yt_put:
                parameters["ya_yt_store_codec"] = self.Parameters.ya_yt_store_codec
                parameters["ya_yt_replace_result"] = self.Parameters.ya_yt_replace_result
                parameters["ya_yt_max_cache_size"] = self.Parameters.ya_yt_max_cache_size

        if self.Parameters.clear_build:
            parameters["semi_clear_build"] = self.Parameters.semi_clear_build

        build_task = sdk2.Task["YA_PACKAGE"](
            self,
            owner=self.owner,
            priority=self.Parameters.priority,
            description="Build RTMR system task {task}={version}".format(
                task=target.name,
                version=rtmr_common.get_version(self),
            ),
            **parameters
        )

        build_task.Requirements.disk_space = self.Parameters.build_disk_space * 1024
        build_task.save()
        build_task.enqueue()

        self.Context.build_tasks.append(build_task.id)
        self.Context.save()

    def on_execute(self):
        with self.memoize_stage.update_version:
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        with self.memoize_stage.start_build(commit_on_entrance=False):
            self.set_info("Start build task(s)")

            if self.Parameters.build_pqout:
                self.build_target(
                    BuildTarget(
                        "pqout",
                        RtmrPqoutPackage))

            if self.Parameters.build_ydbout:
                self.build_target(
                    BuildTarget(
                        "ydbout",
                        RtmrYdboutPackage))

            if self.Parameters.build_solomonout:
                self.build_target(
                    BuildTarget(
                        "solomonout",
                        RtmrSolomonoutPackage))

            self.set_info("Wait build task(s)")

        rtmr_common.wait_tasks(self.Context.build_tasks)
