# -*- coding: utf-8 -*-

import os

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import sdk2
from sandbox.projects.rtmr.resources import RtmrConfigTool, RtmrDeployTool, RtmrDeployDarwinTool, RtmrPushTool, RtmrYFClient


class BuildTarget:

    def __init__(self, name, build_path, resource_type, target_platform=None):
        self.name = name
        self.build_path = build_path
        self.resource_type = resource_type
        self.target_platform = target_platform


class RtmrBuildTools(sdk2.Task):
    """Build RTMR tools"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Build RTMR tools"
        kill_timeout = 3 * 3600  # 3 hours

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )

        build_rtmr_config = sdk2.parameters.Bool(
            "Build rtmr-config tool",
            required=False,
            default_value=True)

        build_rtmr_deploy = sdk2.parameters.Bool(
            "Build rtmr-deploy tool",
            required=False,
            default_value=True)

        build_rtmr_push = sdk2.parameters.Bool(
            "Build rtmr-push tool",
            required=False,
            default_value=True)

        build_yf_client = sdk2.parameters.Bool(
            "Build yf client",
            required=False,
            default_value=True)

    class Context(sdk2.Task.Context):
        build_tasks = list()
        arcadia_branch = None
        arcadia_revision = None

    def build_target(self, target):
        build_task = sdk2.Task["YA_MAKE"](
            self,
            description="Build RTMR tool " + target.name,
            owner=self.owner,
            priority=self.Parameters.priority,
            checkout_arcadia_from_url=rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url),
            arts=os.path.join(target.build_path, target.name),
            targets=target.build_path,
            result_rt=target.resource_type.name,
            result_rd="RTMR tool {name} {version}".format(name=target.name, version=rtmr_common.get_version(self)),
            result_single_file=True,
            checkout_mode="auto",
            build_type="release",
            build_arch="linux",
            target_platform_flags="--target-platform %s" % target.target_platform if target.target_platform else ""
        )
        build_task.save().enqueue()
        self.Context.build_tasks.append(build_task.id)
        self.Context.save()

    def on_execute(self):
        with self.memoize_stage.update_version:
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        with self.memoize_stage.start_build(commit_on_entrance=False):
            self.set_info("Start build task(s)")

            if self.Parameters.build_rtmr_config:
                self.build_target(
                    BuildTarget(
                        "rtmr-config",
                        "rtmapreduce/tools/config",
                        RtmrConfigTool))

            if self.Parameters.build_rtmr_deploy:
                self.build_target(
                    BuildTarget(
                        "rtmr-deploy",
                        "rtmapreduce/tools/deploy",
                        RtmrDeployTool))

            if self.Parameters.build_rtmr_deploy:
                self.build_target(
                    BuildTarget(
                        "rtmr-deploy",
                        "rtmapreduce/tools/deploy",
                        RtmrDeployDarwinTool,
                        "DEFAULT-DARWIN-X86_64"))

            if self.Parameters.build_rtmr_push:
                self.build_target(
                    BuildTarget(
                        "rtmr-push",
                        "rtmapreduce/tools/push",
                        RtmrPushTool))

            if self.Parameters.build_yf_client:
                self.build_target(
                    BuildTarget(
                        "yf",
                        "kikimr/yf/tools/yf_client",
                        RtmrYFClient))

            self.set_info("Wait build task(s)")

        rtmr_common.wait_tasks(self.Context.build_tasks)
