# -*- coding: utf-8 -*-

import logging

from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.resources import RtmrDeployTool
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.projects.rtmr.common as rtmr_common


class RtmrCleanupUsertaskYF(sdk2.Task):
    """Cleanup usertasks deployed to YF"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1 * 1024  # 1Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Cleanup usertasks deployed to YF"
        kill_timeout = 1 * 60 * 60  # 1 hour

        rtmr_deploy_resource = rtmr_common.LastResource(
            "RTMR Deploy Tool",
            resource_type=RtmrDeployTool,
            required=True
        )

        cluster = sdk2.parameters.String("Cluster name", required=True)

        account = sdk2.parameters.String(
            "Name of account to cleanup",
            required=True)

        oauth_token_name = sdk2.parameters.String(
            "Vault secret name with rtmr-deploy OAuth token",
            default="rtmr_deploy_usertask_yf_oauth_token",
            required=True)

    class Context(sdk2.Task.Context):
        rtmr_deploy = None

    def fetch_auth_token(self):
        self.oauth_token = sdk2.Vault.data(self.Parameters.oauth_token_name)

    def update_deploy_tool(self):
        if self.Context.rtmr_deploy is not None:
            return
        rtmr_deploy_resource = self.Parameters.rtmr_deploy_resource
        self.Context.rtmr_deploy = str(sdk2.ResourceData(rtmr_deploy_resource).path)
        self.Context.save()

    def do_cleanup(self):
        cmd = [self.Context.rtmr_deploy,
               "-v", "cleanup",
               "--cluster", self.Parameters.cluster,
               "--account", self.Parameters.account,
               "--oauth-token", self.oauth_token,
               ]

        logging.info("Executing rtmr-deploy with command line: %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise common.errors.TaskError(
                "rtmr-deploy error: {retcode} - {stdout}".format(
                    retcode=proc.returncode, stdout=stdout))

    def on_execute(self):
        self.fetch_auth_token()
        self.update_deploy_tool()

        with self.memoize_stage.cleanup(commit_on_entrance=False):
            self.do_cleanup()
