# -*- coding: utf-8 -*-

import logging
import os

from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.common import get_rtmr_scripts
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp


class RtmrClusterBackup(sdk2.Task):
    """Backup RTMR Cluster"""

    CONFIG_PATH = "rtmapreduce/config/src"
    SCRIPTS_PATH = "rtmapreduce/scripts"

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 5 * 1024  # 5Gb
        environments = [
            environments.PipEnvironment('requests'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Backup RTMR cluster"
        kill_timeout = 24 * 3600
        with sdk2.parameters.String("Cluster name", multiline=True, required=True) as cluster_name:
            _first = True
            for _name in RTMR_CLUSTERS:
                if _first:
                    cluster_name.values[_name] = cluster_name.Value(default=True)
                    _first = False
                else:
                    cluster_name.values[_name] = None
        max_partitions = sdk2.parameters.Integer(
            "Number of simultenious backups on each node",
            required=False,
            default_value=None
        )
        scripts_url = sdk2.parameters.ArcadiaUrl(
            "RTMR Scripts Arcadia url",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )

    def on_execute(self):
        scripts_path = get_rtmr_scripts(self, self.Parameters.scripts_url)
        backup_py = os.path.join(scripts_path, "backup.py")

        cmd = [backup_py, "backup"]
        if self.Parameters.max_partitions is not None:
            cmd.extend(["--max-partitions", str(self.Parameters.max_partitions)])
        cmd.extend(["-c", self.Parameters.cluster_name])
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("backup_py")) as pl:
            proc = sp.Popen(cmd, stdout=pl.stdout, stderr=sp.STDOUT)
            proc.wait()
            if proc.returncode != 0:
                raise common.errors.TaskError("Backup return code: " + str(proc.returncode))

        # Show Stats
        cmd = [backup_py, "info", "-c", self.Parameters.cluster_name]
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            self.set_info("When getting stats we got error code: " + str(proc.returncode))
