# -*- coding: utf-8 -*-

import json
import logging
import os

import sandbox.projects.rtmr.resources as rtmr_resources
import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.RtmrApplyUsertaskConfig import RtmrApplyUsertaskConfig
from sandbox.projects.rtmr.RtmrBuildCommonUserdata import RtmrBuildCommonUserdata
from sandbox.projects.rtmr.RtmrBuildUsertask import RtmrBuildUsertask
from sandbox.projects.rtmr.RtmrDeployZ2 import RtmrDeployZ2
from sandbox.projects.rtmr.RtmrRestartHost import RtmrRestartHost
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.common import wait_tasks, LastResource


class RtmrDeploy(sdk2.Task):
    """Deploy components of RTMR"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Deploy components of RTMR"
        kill_timeout = 3 * 60 * 60

        with sdk2.parameters.String("Cluster name", multiline=True, required=True) as cluster_name:
            _first = True
            for _name in RTMR_CLUSTERS:
                if _first:
                    cluster_name.values[_name] = cluster_name.Value(default=True)
                    _first = False
                else:
                    cluster_name.values[_name] = None

        with sdk2.parameters.Group("Components") as components_block:
            daemons_task = sdk2.parameters.Task("Daemons build task", task_type="YA_PACKAGE")
            usertask_task = sdk2.parameters.Task("Usertask build task", task_type=RtmrBuildUsertask)
            userdata_task = sdk2.parameters.Task("Userdata build task", task_type=RtmrBuildCommonUserdata)

            add_last_userdata_resource = sdk2.parameters.Bool(
                "Add last common userdata resource if userdata build task is not set",
                default_value=False)

            with sdk2.parameters.RadioGroup("Filtering options") as filter:
                filter.values.none = filter.Value("none", default=True)
                filter.values.include = filter.Value("include")
                filter.values.exclude = filter.Value("exclude")

            with filter.value["include"]:
                task_ids = sdk2.parameters.String(
                    "Comma separated list of task ids which packages need to be "
                    "deployed (empty for all packages from usertask build task)")

            with filter.value["exclude"]:
                with sdk2.parameters.RadioGroup("Exclude target") as exclude_target:
                    exclude_target.values.accounts = exclude_target.Value("accounts", default=True)
                    exclude_target.values.tasks = exclude_target.Value("tasks")

                with exclude_target.value["accounts"]:
                    exclude_accounts = sdk2.parameters.String(
                        "Comma separated list of accounts which tasks' binaries and "
                        "config should be excluded from deploy",
                        required=False)

                with exclude_target.value["tasks"]:
                    exclude_tasks = sdk2.parameters.String(
                        "Comma separated list of task ids which binaries and config "
                        "should be excluded from deploy",
                        required=False)

                with sdk2.parameters.RadioGroup("Exclude packages mode") as exclude_packages_mode:
                    exclude_packages_mode.values.strict = exclude_packages_mode.Value("strict", default=True)
                    exclude_packages_mode.values.tasks = exclude_packages_mode.Value("relaxed")

        rtmr_config_resource = LastResource(
            "RTMR Config Tool",
            resource_type=rtmr_resources.RtmrConfigTool,
            required=True
        )

        with sdk2.parameters.Group("Deploy settings") as z2_settings:
            apply_config = sdk2.parameters.Bool("Apply usertask config", default_value=True)
            restart_host = sdk2.parameters.Bool("Restart host after deploy", default_value=True)
            with restart_host.value[True]:
                with sdk2.parameters.Group("Restart host auth settings") as auth_block:
                    host_secret_name = sdk2.parameters.String("Vault secret name with SSH key")
                    host_secret_owner = sdk2.parameters.String("Vault secret owner")
                    host_remote_user = sdk2.parameters.String("Remote username")
            retry_time = sdk2.parameters.Integer("Z2 retry time (seconds)", required=True, default_value=3 * 60)
            retry_limit = sdk2.parameters.Integer("Z2 retry limit", required=True, default_value=5)
            send_error_email = sdk2.parameters.String("Z2 send error email to")

    class Context(sdk2.Task.Context):
        rtmr_gencfg_path = None
        rtmr_configs_path = None
        deploy_taskids = list()
        packages = dict()

    def collect_packages(self, build_task, resource_type=None, name_filter=None):
        packages = dict()
        tasks = [build_task]
        build_subtasks = build_task.find()
        tasks.extend(build_subtasks)
        build_sub_subtasks = []
        for task in build_subtasks:
            subtasks = task.find()
            build_sub_subtasks.extend(subtasks)
            tasks.extend(subtasks)
        for task in build_sub_subtasks:
            tasks.extend(task.find())
        for task in tasks:
            for resource in sdk2.Resource.find(task=task).limit(0):
                if resource_type is not None and resource.type != resource_type:
                    continue
                if name_filter is not None and \
                        hasattr(resource, "resource_name") and \
                        resource.resource_name not in name_filter:
                    continue
                if hasattr(resource, "resource_version"):
                    packages[resource.resource_name] = resource.resource_version
        return packages

    def packages_info(self, resources):
        return "\n".join([name + "=" + version for (name, version) in resources.iteritems()])

    def show_packages(self, resources):
        self.set_info("Packages:\n" + self.packages_info(resources))

    def get_z2_secret_name(self):
        return "rtmr-z2-" + self.Parameters.cluster_name

    def deploy_z2(self, packages):
        params = dict(
            secret_name=self.get_z2_secret_name(),
            packages=packages,
            cluster_name=self.Parameters.cluster_name,
        )
        if self.Parameters.send_error_email:
            params["send_error_email"] = self.Parameters.send_error_email
        if self.Parameters.retry_limit:
            params["retry_limit"] = self.Parameters.retry_limit
        if self.Parameters.retry_time:
            params["retry_time"] = self.Parameters.retry_time
        if self.Parameters.restart_host:
            params["restart_host"] = self.Parameters.restart_host
            if self.Parameters.host_secret_name:
                params["host_secret_name"] = self.Parameters.host_secret_name
            if self.Parameters.host_secret_owner:
                params["host_secret_owner"] = self.Parameters.host_secret_owner
            if self.Parameters.host_remote_user:
                params["host_remote_user"] = self.Parameters.host_remote_user

        task = RtmrDeployZ2(
            self,
            description="Deploy to {} packages:\n{}".format(self.Parameters.cluster_name, self.packages_info(packages)),
            priority=self.Parameters.priority,
            **params
        )
        task.save().enqueue()
        self.Context.deploy_taskids.append(task.id)
        self.Context.save()

    def deploy_daemons(self):
        self.set_info("Deploy daemons")
        packages = self.collect_packages(self.Parameters.daemons_task, resource_type=rtmr_resources.RtmrReleaseDeb)
        if len(packages) == 0:
            raise common.errors.TaskError("Not found daemons packages")
        self.show_packages(packages)
        self.deploy_z2(packages)

    def deploy_userdata(self):
        self.set_info("Deploy userdata packages")
        self.show_packages(self.Context.packages)
        self.deploy_z2(self.Context.packages)

    def collect_userdata_package(self):
        packages = self.collect_packages(
            self.Parameters.userdata_task,
            resource_type=rtmr_resources.RtmrUserdataDeb
        )
        if len(packages) == 0:
            raise common.errors.TaskError("Not found common userdata package")
        self.Context.packages.update(packages)
        self.Context.save()

    def collect_userdata_package_from_resource(self):
        resource = sdk2.Resource.find(
            resource_type=rtmr_resources.RtmrUserdataDeb,
            state=common.types.resource.State.READY)\
            .order(-sdk2.Resource.id)\
            .first()
        if hasattr(resource, "resource_version"):
            self.Context.packages.update({resource.resource_name: resource.resource_version})
            self.Context.save()

    def collect_usertask_packages(self):
        resource = rtmr_common.find_packageconfig(self.Parameters.usertask_task)
        if resource is None:
            raise common.errors.TaskError(
                "Could not find package config resource")

        pkg_config_path = str(sdk2.ResourceData(resource).path)
        with open(pkg_config_path, "r") as fd:
            try:
                pkg_config = json.load(fd)
            except ValueError:
                raise common.errors.TaskError("Package config has invalid json")
        if self.Parameters.cluster_name not in pkg_config:
            raise common.errors.TaskError("Not found packages for cluster")
        packages_filter = set(pkg_config[self.Parameters.cluster_name])
        packages_filter = self.filter_usertask_packages(packages_filter)
        logging.info("Filtered cluster packages %r", packages_filter)
        packages = self.collect_packages(
            self.Parameters.usertask_task,
            resource_type=rtmr_resources.RtmrUsertaskDeb,
            name_filter=packages_filter,
        )
        if len(packages) == 0:
            raise common.errors.TaskError("Not found usertask package resources")
        self.Context.packages.update(packages)
        self.Context.save()

    def list_include_tasks(self):
        if self.Parameters.filter != "include":
            return []

        if not self.Parameters.task_ids:
            return []

        task_ids = self.Parameters.task_ids.split(',')
        return filter(bool, [t.strip() for t in task_ids])

    def list_exclude_tasks(self):
        if self.Parameters.filter != "exclude":
            return []

        if self.Parameters.exclude_tasks:
            task_ids = self.Parameters.exclude_tasks.split(',')
            return filter(bool, [t.strip() for t in task_ids])

        if self.Parameters.exclude_accounts:
            return rtmr_common.list_tasks_for_account(
                self,
                self.Parameters.exclude_accounts,
                self.Parameters.cluster_name,
                self.get_arcadia_url())

        return []

    def filter_usertask_packages(self, packages):
        include_tasks = self.list_include_tasks()
        exclude_tasks = self.list_exclude_tasks()

        if not include_tasks and not exclude_tasks:
            return packages

        if include_tasks and exclude_tasks:
            raise common.errors.TaskError(
                "Both task ids and exclude tasks cannot be specified")

        arcadia_url = self.get_arcadia_url()

        if include_tasks:
            return rtmr_common.list_packages_for_tasks(
                self,
                include_tasks,
                self.Parameters.cluster_name,
                arcadia_url)

        all_tasks = rtmr_common.list_all_tasks(
            self,
            self.Parameters.cluster_name,
            arcadia_url)

        include_tasks = all_tasks.difference(exclude_tasks)

        include_packages = rtmr_common.list_packages_for_tasks(
            self,
            include_tasks,
            self.Parameters.cluster_name,
            arcadia_url)

        if self.Parameters.exclude_packages_mode == "relaxed":
            return include_packages

        # Need to ensure that include_packages does not contain any packages
        # corresponding to tasks being excluded (except well known packages
        # such as out operations packages and common userdata)
        exclude_packages = rtmr_common.list_packages_for_tasks(
            self,
            exclude_tasks,
            self.Parameters.cluster_name,
            arcadia_url)

        common_packages = include_packages.intersection(exclude_packages)
        # remove well known packages
        common_packages.discard("yandex-search-rtmr-common-userdata")
        common_packages.discard("yandex-search-rtmr-task-pqout")
        common_packages.discard("yandex-search-rtmr-task-solomonout")
        common_packages.discard("yandex-search-rtmr-task-rtmrout")
        common_packages.discard("yandex-search-rtmr-task-ydbout")

        if len(common_packages) != 0:
            affected_tasks = rtmr_common.list_tasks_for_packages(
                self,
                common_packages,
                self.Parameters.cluster_name,
                arcadia_url)

            non_excluded_affected_tasks = affected_tasks.difference(
                exclude_tasks)

            raise common.errors.TaskError(
                "Cannot exclude packages from deploy: the following packages \
                belong to other tasks which are not being excluded: {packages}\n \
                Affected tasks: {tasks}".format(
                    packages=", ".join(common_packages),
                    tasks=", ".join(non_excluded_affected_tasks)))

        return include_packages

    def get_cluster_config_path(self):
        return os.path.join(
            rtmr_common.get_rtmr_configs(
                self,
                self.Parameters.usertask_task.Parameters.arcadia_url),
            self.Parameters.cluster_name + ".cfg"
        )

    def get_arcadia_url(self):
        if self.Parameters.usertask_task.type == "RTMR_USER_RELEASE" or \
                self.Parameters.usertask_task.type == "RTMR_RELEASE":
            package_task = self.Parameters.usertask_task.find(
                task_type=sdk2.Task["RTMR_BUILD_USERTASK"]).first()
        else:
            package_task = self.Parameters.usertask_task

        return package_task.Parameters.arcadia_url

    def restart_host(self):
        self.set_info("Restart host process")
        parameters = dict(cluster_name=self.Parameters.cluster_name)
        if self.Parameters.host_secret_name:
            parameters["secret_name"] = self.Parameters.host_secret_name
        if self.Parameters.host_secret_owner:
            parameters["secret_owner"] = self.Parameters.host_secret_owner
        if self.Parameters.host_remote_user:
            parameters["remote_user"] = self.Parameters.host_remote_user
        task = RtmrRestartHost(
            self,
            description="Restart host on " + self.Parameters.cluster_name,
            priority=self.Parameters.priority,
            **parameters
        )
        task.save().enqueue()
        self.Context.deploy_taskids.append(task.id)
        self.Context.save()

    def apply_config(self):
        self.set_info("Apply usertask config")

        exclude_tasks = self.list_exclude_tasks()
        if exclude_tasks:
            exclude_tasks = ",".join(exclude_tasks)

        task = RtmrApplyUsertaskConfig(
            self,
            description="Apply usertasks config on cluster " + self.Parameters.cluster_name,
            priority=self.Parameters.priority,
            cluster_name=self.Parameters.cluster_name,
            config_task=self.Parameters.usertask_task,
            rtmr_config_resource=self.Parameters.rtmr_config_resource,
            task_ids=self.Parameters.task_ids,
            exclude_tasks=exclude_tasks or None,
        )
        task.save().enqueue()
        self.Context.deploy_taskids.append(task.id)
        self.Context.save()

    def on_execute(self):
        if self.Parameters.daemons_task is None and \
           self.Parameters.usertask_task is None and \
           self.Parameters.userdata_task is None:
            raise common.errors.TaskError("Need specify source tasks")

        if self.Parameters.daemons_task is not None:
            with self.memoize_stage.deploy_daemons(commit_on_entrance=False):
                self.deploy_daemons()
            with self.memoize_stage.wait_deploy_daemons(commit_on_entrance=False, commit_on_wait=False):
                wait_tasks(self.Context.deploy_taskids)
                self.Context.deploy_taskids = list()
                self.Context.save()

        if self.Parameters.userdata_task is not None:
            with self.memoize_stage.collect_userdata_resources(commit_on_entrance=False):
                self.collect_userdata_package()
        elif self.Parameters.add_last_userdata_resource:
            with self.memoize_stage.collect_userdata_resources(commit_on_entrance=False):
                self.collect_userdata_package_from_resource()

        if self.Parameters.usertask_task is not None:
            if self.Parameters.task_ids is not None:
                with self.memoize_stage.build_gencfg(commit_on_entrance=False):
                    rtmr_common.get_rtmr_gencfg(
                        self,
                        self.Parameters.usertask_task.Parameters.arcadia_url)

            with self.memoize_stage.collect_usertasks_resources(commit_on_entrance=False):
                self.collect_usertask_packages()

        if len(self.Context.packages) > 0:
            with self.memoize_stage.deploy_userdata(commit_on_entrance=False):
                self.deploy_userdata()

        with self.memoize_stage.wait_deploy_userdata(commit_on_entrance=False, commit_on_wait=False):
            wait_tasks(self.Context.deploy_taskids)
            self.Context.deploy_taskids = list()
            self.Context.save()

        if self.Parameters.usertask_task is not None and self.Parameters.apply_config:
            with self.memoize_stage.apply_config(commit_on_entrance=False):
                self.apply_config()
            with self.memoize_stage.wait_apply_config(commit_on_entrance=False, commit_on_wait=False):
                wait_tasks(self.Context.deploy_taskids)
                self.Context.deploy_taskids = list()
                self.Context.save()

        if self.Parameters.restart_host:
            with self.memoize_stage.restart_host(commit_on_entrance=False):
                self.restart_host()

            wait_tasks(self.Context.deploy_taskids)
