# -*- coding: utf-8 -*-

import logging
import os

from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.resources import RtmrDeployTool
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.projects.rtmr.common as rtmr_common


class RtmrDeployGraphYF(sdk2.Task):
    """Deploy graph for YF"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Deploy graph for YF"
        kill_timeout = 3 * 60 * 60  # 3 hours

        rtmr_deploy_resource = rtmr_common.LastResource(
            "RTMR Deploy Tool",
            resource_type=RtmrDeployTool,
            required=True
        )

        cluster = sdk2.parameters.String("Cluster name", required=True)

        graph = sdk2.parameters.String(
            "Name of graph to deploy",
            required=True)

        cleanup_missing = sdk2.parameters.Bool(
            "Delete tasks not present in the generated config",
            default=False)

        build_task = sdk2.parameters.String(
            "RTMR_GRAPH_RELEASE task used to build artifacts",
            required=True)

        slots = sdk2.parameters.Integer(
            "Number of slots for YF function",
            default=None)

        oauth_token_name = sdk2.parameters.String(
            "Vault secret name with rtmr-deploy OAuth token",
            default="rtmr_deploy_usertask_yf_oauth_token",
            required=True)

    class Context(sdk2.Task.Context):
        rtmr_deploy = None
        rtmr_gencfg_path = None
        config_backup_resource = None
        rollback_task = None
        cleanup_task = None
        excluded_task_ids = list()

    def fetch_auth_token(self):
        self.oauth_token = sdk2.Vault.data(self.Parameters.oauth_token_name)

    def update_deploy_tool(self):
        if self.Context.rtmr_deploy is not None:
            return
        rtmr_deploy_resource = self.Parameters.rtmr_deploy_resource
        self.Context.rtmr_deploy = str(sdk2.ResourceData(rtmr_deploy_resource).path)
        self.Context.save()

    def get_cluster_config_path(self, cluster_name, arcadia_url):
        return os.path.join(
            rtmr_common.get_rtmr_configs(self, arcadia_url),
            cluster_name + ".cfg"
        )

    def get_confirmation_arg(self):
        if self.Parameters.cluster not in ["rtmr-vla", "rtmr-sas"]:
            confirmation = None
        else:
            confirmation = "--yes-please-destroy-production-cluster"

        return confirmation

    def do_deploy(self):
        cmd = [self.Context.rtmr_deploy,
               "-v",
               "release",
               "--cluster", self.Parameters.cluster,
               "--account", self.Parameters.graph,
               "--build-task", self.Parameters.build_task,
               "--owner", self.owner,
               "--oauth-token", self.oauth_token,
               ]

        confirmation = self.get_confirmation_arg()
        if confirmation:
            cmd.append(confirmation)

        if self.Parameters.slots:
            cmd.extend(["--slots", str(self.Parameters.slots)])

        if not self.Parameters.cleanup_missing:
            cmd.extend(["--cleanup-missing", "False"])

        logging.info("Executing rtmr-deploy with command line: %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise common.errors.TaskError(
                "rtmr-deploy error: {retcode} - {stdout}".format(
                    retcode=proc.returncode, stdout=stdout))

    def get_arcadia_url(self):
        build_task = sdk2.Task[self.Parameters.build_task]

        package_task = build_task.find(
            task_type=sdk2.Task["RTMR_BUILD_GRAPH"]).first()

        return package_task.Parameters.arcadia_url

    def on_execute(self):
        self.fetch_auth_token()
        self.update_deploy_tool()

        with self.memoize_stage.deploy(commit_on_entrance=False):
            self.do_deploy()
