# -*- coding: utf-8 -*-

import logging
import os
import shutil

from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.resources import RtmrDeployTool, RtmrUsertaskConfig
from sandbox.projects.rtmr.RtmrCleanupUsertaskYF import RtmrCleanupUsertaskYF
from sandbox.projects.rtmr.RtmrRollbackUsertaskYF import RtmrRollbackUsertaskYF
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.projects.rtmr.common as rtmr_common


class RtmrDeployUsertaskYF(sdk2.Task):
    """Deploy usertasks for YF"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Deploy usertasks for YF"
        kill_timeout = 3 * 60 * 60  # 3 hours

        rtmr_deploy_resource = rtmr_common.LastResource(
            "RTMR Deploy Tool",
            resource_type=RtmrDeployTool,
            required=True
        )

        cluster = sdk2.parameters.String("Cluster name", required=True)

        account = sdk2.parameters.String(
            "Name of account to deploy",
            required=True)

        update_previous = sdk2.parameters.Bool(
            "Update artifacts of previous version instead of full release",
            default=False)

        with sdk2.parameters.RadioGroup("Deploy target") as deploy_target:
            deploy_target.values.account = deploy_target.Value("account", default=True)
            deploy_target.values.tasks = deploy_target.Value("tasks")

        with deploy_target.value["account"]:
            cleanup_missing = sdk2.parameters.Bool(
                "Delete tasks not present in the generated config",
                default=False)

            with sdk2.parameters.RadioGroup("Exclude target") as exclude_target:
                exclude_target.values.accounts = exclude_target.Value("accounts", default=True)
                exclude_target.values.tasks = exclude_target.Value("tasks")

            with exclude_target.value["accounts"]:
                exclude_accounts = sdk2.parameters.String(
                    "Comma separated list of accounts which tasks' binaries and "
                    "config should be excluded from deploy",
                    required=False)

            with exclude_target.value["tasks"]:
                exclude_tasks = sdk2.parameters.String(
                    "Comma separated list of task ids which binaries and config "
                    "should be excluded from deploy",
                    required=False)

            with sdk2.parameters.RadioGroup("Exclude packages mode") as exclude_packages_mode:
                exclude_packages_mode.values.strict = exclude_packages_mode.Value("strict", default=True)
                exclude_packages_mode.values.relaxed = exclude_packages_mode.Value("relaxed")

        with deploy_target.value["tasks"]:
            task_ids = sdk2.parameters.String(
                "Comma separated list of task ids to deploy "
                "(empty to deploy all built tasks)")

        build_task = sdk2.parameters.String(
            "RTMR_USER_RELEASE task used to build artifacts",
            required=True)

        cleanup_first = sdk2.parameters.Bool(
            "Run rtmr-deploy cleanup before deploying the new version",
            default=True)

        cleanup_last = sdk2.parameters.Bool(
            "Run rtmr-deploy cleanup after deploying the new version",
            default=False)

        switch = sdk2.parameters.Bool(
            "Switch execution to new version",
            default=True)

        slots = sdk2.parameters.Integer(
            "Number of slots for YF function",
            default=None)

        oauth_token_name = sdk2.parameters.String(
            "Vault secret name with rtmr-deploy OAuth token",
            default="rtmr_deploy_usertask_yf_oauth_token",
            required=True)

        notification_logins = sdk2.parameters.String(
            "Comma separated list of staff logins for crash notifications (by e-mail)",
            required=False,
            default=None)

    class Context(sdk2.Task.Context):
        rtmr_deploy = None
        rtmr_gencfg_path = None
        previous_version = None
        config_backup_resource = None
        rollback_task = None
        cleanup_task = None
        excluded_task_ids = list()

    def fetch_auth_token(self):
        self.oauth_token = sdk2.Vault.data(self.Parameters.oauth_token_name)

    def update_deploy_tool(self):
        if self.Context.rtmr_deploy is not None:
            return
        rtmr_deploy_resource = self.Parameters.rtmr_deploy_resource
        self.Context.rtmr_deploy = str(sdk2.ResourceData(rtmr_deploy_resource).path)
        self.Context.save()

    def update_excluded_task_ids(self):
        if self.Parameters.deploy_target != "account":
            return

        if self.Parameters.exclude_target == "tasks":
            exclude_tasks = self.Parameters.exclude_tasks.split(",")
            exclude_tasks = filter(bool, [t.strip() for t in exclude_tasks])
            self.Context.excluded_task_ids = exclude_tasks
            self.Context.save()
            return

        if not self.Parameters.exclude_accounts:
            return

        arcadia_url = self.get_arcadia_url()

        excluded_task_ids = rtmr_common.list_tasks_for_account(
            self,
            self.Parameters.exclude_accounts,
            self.Parameters.cluster,
            arcadia_url)

        self.Context.excluded_task_ids.extend(excluded_task_ids)
        self.Context.save()

    def fetch_previous_version(self):
        if self.Context.previous_version is not None:
            return

        try:
            current_version = self.get_current_version()
        except Exception as e:
            logging.info("Failed to find current version: %r", e)
            self.Context.previous_version = None
        else:
            self.Context.previous_version = current_version

        self.Context.save()

    def get_cluster_config_path(self, cluster_name, arcadia_url):
        return os.path.join(
            rtmr_common.get_rtmr_configs(self, arcadia_url),
            cluster_name + ".cfg"
        )

    def get_confirmation_arg(self):
        if not self.Parameters.switch or \
                self.Parameters.cluster not in ["rtmr-man4", "rtmr-vla", "rtmr-sas"]:
            confirmation = None
        else:
            confirmation = "--yes-please-destroy-production-cluster"

        return confirmation

    def do_cleanup(self):
        cmd = [self.Context.rtmr_deploy,
               "-v", "cleanup",
               "--cluster", self.Parameters.cluster,
               "--account", self.Parameters.account,
               "--oauth-token", self.oauth_token,
               ]

        logging.info("Executing rtmr-deploy with command line: %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise common.errors.TaskError(
                "rtmr-deploy error: {retcode} - {stdout}".format(
                    retcode=proc.returncode, stdout=stdout))

    def do_deploy(self):
        if self.Parameters.update_previous:
            command = "update"
        else:
            command = "release"

        cmd = [self.Context.rtmr_deploy,
               "-v",
               command,
               "--cluster", self.Parameters.cluster,
               "--account", self.Parameters.account,
               "--build-task", self.Parameters.build_task,
               "--owner", self.owner,
               "--oauth-token", self.oauth_token,
               ]

        if self.Parameters.notification_logins:
            notification_logins = self.Parameters.notification_logins.split(",")
            for login in notification_logins:
                cmd.extend(["--notification-login", login.strip()])

        if self.Parameters.deploy_target == "account" and \
                self.Context.excluded_task_ids:
            for task_id in self.Context.excluded_task_ids:
                cmd.extend(["--config-exclude-task", task_id])

            included_packages = self.list_packages_to_include()
            if included_packages:
                artifacts_filter = "(" + "|".join(included_packages) + ")"
                cmd.extend(["--artifacts-filter", artifacts_filter])
                cmd.extend(["--artifacts-filter-type", "in"])
        elif self.Parameters.deploy_target == "tasks":
            task_ids = self.Parameters.task_ids.split(",")
            task_ids = set(filter(bool, [t.strip() for t in task_ids]))

            for task_id in task_ids:
                cmd.extend(["--config-include-task", task_id])

            arcadia_url = self.get_arcadia_url()

            filtered_packages = rtmr_common.list_packages_for_tasks(
                self,
                task_ids,
                self.Parameters.cluster,
                arcadia_url)

            if filtered_packages:
                artifacts_filter = "(" + "|".join(filtered_packages) + ")"
                cmd.extend(["--artifacts-filter", artifacts_filter])
                cmd.extend(["--artifacts-filter-type", "in"])

        confirmation = self.get_confirmation_arg()
        if confirmation:
            cmd.append(confirmation)

        if self.Parameters.slots:
            cmd.extend(["--slots", str(self.Parameters.slots)])

        # for rtmr-deploy update --cleanup-missing has no arg and defaults to False
        # but for rtmr-deploy release it has an arg and defaults to True
        if self.Parameters.update_previous:
            if self.Parameters.cleanup_missing:
                cmd.append("--cleanup-missing")
        elif not self.Parameters.cleanup_missing:
            cmd.extend(["--cleanup-missing", "False"])

        if not self.Parameters.switch:
            if self.Parameters.update_previous:
                cmd.extend(["--with-config", "False"])
            else:
                cmd.append("--dry-run")

        logging.info("Executing rtmr-deploy with command line: %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise common.errors.TaskError(
                "rtmr-deploy error: {retcode} - {stdout}".format(
                    retcode=proc.returncode, stdout=stdout))

    def get_arcadia_url(self):
        build_task = sdk2.Task[self.Parameters.build_task]

        package_task = build_task.find(
            task_type=sdk2.Task["RTMR_BUILD_USERTASK"]).first()

        return package_task.Parameters.arcadia_url

    def list_packages_to_include(self):
        if self.Parameters.deploy_target != "account":
            return None

        arcadia_url = self.get_arcadia_url()

        if self.Parameters.exclude_target == "accounts":
            exclude_accounts = self.Parameters.exclude_accounts.split(",")
            exclude_accounts = set(filter(bool, [a.strip() for a in exclude_accounts]))
            if not exclude_accounts:
                return None

            all_accounts = rtmr_common.list_all_accounts(
                self,
                self.Parameters.cluster,
                arcadia_url)

            include_accounts = all_accounts.difference(exclude_accounts)
            include_tasks = rtmr_common.list_tasks_for_account(
                self,
                ",".join(include_accounts),
                self.Parameters.cluster,
                arcadia_url)

            logging.info(
                "Exclude accounts: {exclude_accounts}\n\n".format(
                    exclude_accounts=", ".join(exclude_accounts)) +
                "Include accounts: {include_accounts}\n\n".format(
                    include_accounts=", ".join(include_accounts)) +
                "Include tasks: {include_tasks}".format(
                    include_tasks=", ".join(include_tasks)))
        elif self.Parameters.exclude_target == "tasks":
            exclude_tasks = self.Parameters.exclude_tasks.split(",")
            exclude_tasks = set(filter(bool, [t.strip() for t in exclude_tasks]))

            if not exclude_tasks:
                return None

            all_tasks = rtmr_common.list_all_tasks(
                self,
                self.Parameters.cluster,
                arcadia_url)

            include_tasks = all_tasks.difference(exclude_tasks)

            logging.info(
                "Exclude tasks: {exclude_tasks}\n\n".format(
                    exclude_tasks=", ".join(exclude_tasks)) +
                "All tasks: {all_tasks}\n\n".format(
                    all_tasks=", ".join(all_tasks)) +
                "Include tasks: {include_tasks}".format(
                    include_tasks=", ".join(include_tasks)))

        if not include_tasks:
            return None

        include_packages = rtmr_common.list_packages_for_tasks(
            self,
            include_tasks,
            self.Parameters.cluster,
            arcadia_url)

        # common userdata needs to be added as it is not assigned to any task
        include_packages.add("yandex-search-rtmr-common-userdata")

        # HACK: ydbout needs to be within "default" account even though no tasks
        # from it are currently linked to it
        if self.Parameters.account == "default":
            include_packages.add("yandex-search-rtmr-task-ydbout")

        logging.info("Include packages: {}".format(", ".join(include_packages)))

        if self.Parameters.exclude_packages_mode == "relaxed":
            return include_packages

        # Need to ensure that include_packages does not contain any packages
        # corresponding to tasks being excluded (except well known packages
        # such as out operations packages and common userdata)
        exclude_packages = rtmr_common.list_packages_for_tasks(
            self,
            self.Context.excluded_task_ids,
            self.Parameters.cluster,
            arcadia_url)

        common_packages = include_packages.intersection(exclude_packages)
        # remove well known packages
        common_packages.discard("yandex-search-rtmr-common-userdata")
        common_packages.discard("yandex-search-rtmr-task-pqout")
        common_packages.discard("yandex-search-rtmr-task-solomonout")
        common_packages.discard("yandex-search-rtmr-task-rtmrout")
        common_packages.discard("yandex-search-rtmr-task-ydbout")

        if len(common_packages) != 0:
            affected_tasks = rtmr_common.list_tasks_for_packages(
                self,
                common_packages,
                self.Parameters.cluster,
                arcadia_url)

            non_excluded_affected_tasks = affected_tasks.difference(
                self.Context.excluded_task_ids)

            raise common.errors.TaskError(
                "Cannot exclude packages from deploy: the following packages \
                belong to other tasks which are not being excluded: {packages}\n \
                Affected tasks: {tasks}".format(
                    packages=", ".join(common_packages),
                    tasks=", ".join(non_excluded_affected_tasks)))

        return include_packages

    def get_current_version(self):
        cmd = [self.Context.rtmr_deploy,
               "-l", "list_versions_log.txt",
               "list-versions",
               "--cluster", self.Parameters.cluster,
               "--account", self.Parameters.account,
               "--oauth-token", self.oauth_token,
               ]

        logging.info("Executing rtmr-deploy with command line: %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if proc.returncode != 0:
            raise common.errors.TaskError(
                "rtmr-deploy return code: " + str(proc.returncode))

        version = None
        lines = stdout.splitlines()
        for line in lines:
            line = line.strip()
            if not line.startswith("*"):
                continue

            tokens = line.rsplit(".", 1)
            if len(tokens) != 2 or not tokens[1].isdigit():
                logging.warn("skipping invalid function {name} in version discovery".format(
                    name=line))
                continue
            version = tokens[1]
            break

        if version is None:
            if len(lines) == 0:
                # there are no existing versions
                return None

            raise common.errors.TaskError(
                "Unable to find current version")

        if not version.isdigit():
            raise common.errors.TaskError(
                "Failed to extract version from YF function name")

        return version

    def save_config_backup_to_resource(self, version):
        if self.Context.config_backup_resource is not None:
            return sdk2.Resource[self.Context.config_backup_resource]

        resource = RtmrUsertaskConfig(
            self,
            "RTMR tasks config for account {account} and cluster {cluster}: version = {version}".format(
                account=self.Parameters.account,
                cluster=self.Parameters.cluster,
                version=version,
            ),
            "taskconfig-{}-{}.json".format(self.Parameters.account, self.Parameters.cluster),
            cluster=self.Parameters.cluster,
        )
        resource_data = sdk2.ResourceData(resource)
        dst_path = str(resource_data.path.absolute())

        src_path = os.environ.get("HOME")
        if not src_path:
            src_path = "/tmp"

        src_path = os.path.join(
            src_path,
            ".rtmr-deploy",
            "config-backups",
            self.Parameters.cluster,
            self.Parameters.account + "." + str(version) + ".json")

        shutil.copyfile(src_path, dst_path)
        resource_data.ready()

        self.Context.config_backup_resource = resource.id
        self.Context.save()

        return resource

    def make_rollback_task(self, previous_version, config_backup):
        if self.Context.rollback_task is not None:
            return

        task = RtmrRollbackUsertaskYF(
            self,
            description="Rollback account {account} to version {previous_version} at {cluster}".format(
                account=self.Parameters.account,
                previous_version=previous_version,
                cluster=self.Parameters.cluster),
            priority=self.Parameters.priority,
            rtmr_deploy_resource=self.Parameters.rtmr_deploy_resource,
            cluster=self.Parameters.cluster,
            account=self.Parameters.account,
            config_backup_resource=config_backup,
            rollback_version=previous_version,
            oauth_token_name=self.Parameters.oauth_token_name)

        task.save()

        self.Context.rollback_task = task.id
        self.Context.save()

    def make_cleanup_task(self):
        if self.Context.cleanup_task is not None:
            return

        task = RtmrCleanupUsertaskYF(
            self,
            description="Cleanup YF usertasks from account {account} at {cluster}".format(
                account=self.Parameters.account,
                cluster=self.Parameters.cluster),
            priority=self.Parameters.priority,
            rtmr_deploy_resource=self.Parameters.rtmr_deploy_resource,
            cluster=self.Parameters.cluster,
            account=self.Parameters.account,
            oauth_token_name=self.Parameters.oauth_token_name)

        task.save()

        self.Context.cleanup_task = task.id
        self.Context.save()

    def on_execute(self):
        self.fetch_auth_token()
        self.update_deploy_tool()

        with self.memoize_stage.update_excluded_tasks(commit_on_entrance=False):
            self.update_excluded_task_ids()

        if self.Parameters.cleanup_first:
            with self.memoize_stage.cleanup_first(commit_on_entrance=False):
                self.do_cleanup()

        self.fetch_previous_version()

        with self.memoize_stage.deploy(commit_on_entrance=False):
            self.do_deploy()

        if self.Parameters.cleanup_last:
            with self.memoize_stage.cleanup_last(commit_on_entrance=False):
                self.do_cleanup()
        elif self.Context.previous_version is not None:
            self.make_cleanup_task()

            if self.Parameters.switch:
                config_backup = self.save_config_backup_to_resource(self.Context.previous_version)
                self.make_rollback_task(self.Context.previous_version, config_backup)
