# -*- coding: utf-8 -*-

import json
import logging
import time
import re

import requests

from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS, RtmrClustersInfo
from sandbox.projects.rtmr.common import retry, notify


class Z2TemporaryError(Exception):
    pass


class RtmrDeployZ2(sdk2.Task):
    """Update Z2 configuration"""
    name = "RTMR_DEPLOY_Z2"
    Z2_BASE_URL = "https://z2.yandex-team.ru"
    UNKNOWN_ITEM_RE = re.compile(
        r'java\.lang\.IllegalArgumentException: item with id \'([^\']+)\'',
        re.MULTILINE | re.DOTALL
    )
    INPROGRESS_ERR = "java.lang.IllegalStateException: Update is already in progress"

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Update Z2 configuration"
        kill_timeout = 2 * 60 * 60  # 2 hours

        with sdk2.parameters.String("Cluster name", multiline=True, required=True) as cluster_name:
            _first = True
            for _name in RTMR_CLUSTERS:
                if _first:
                    cluster_name.values[_name] = cluster_name.Value(default=True)
                    _first = False
                else:
                    cluster_name.values[_name] = None

        packages = sdk2.parameters.Dict("Packages")
        clean = sdk2.parameters.Bool("Clean after update", default_value=True)
        secret_name = sdk2.parameters.String("Vault secret name with Z2 key")
        retry_time = sdk2.parameters.Integer(
            "Retry time (seconds)",
            required=True,
            default_value=3 * 60
        )
        retry_limit = sdk2.parameters.Integer("Retry limit", required=True, default_value=5)
        send_error_email = sdk2.parameters.String("Send error email to")

    class Context(sdk2.Task.Context):
        try_no = 0
        z2_configs = None

    def show_packages(self):
        msg = ["Packages:"]
        for name, ver in self.Parameters.packages.iteritems():
            msg.append(name + "=" + ver)
        self.set_info("\n".join(msg))

    def get_z2_items(self):
        return [dict(name=name, version=ver) for name, ver in self.Parameters.packages.iteritems()]

    def get_z2_config(self, force=False):
        if self.Context.z2_configs is None or force:
            self.Context.z2_configs = list(RtmrClustersInfo().clusters[self.Parameters.cluster_name].z2_groups)
        return self.Context.z2_configs[0]

    def get_z2_base_request(self):
        return dict(
            apiKey=sdk2.Vault.data(self.Parameters.secret_name),
            configId=self.get_z2_config()
        )

    def on_execute(self):
        with self.memoize_stage.show_packages:
            self.show_packages()

        with self.memoize_stage.update_z2_groups:
            self.get_z2_config()  # update z2 groups list

        with self.memoize_stage.update(commit_on_entrance=False, commit_on_wait=False):
            while len(self.Context.z2_configs) > 0:
                z2_config = self.get_z2_config()
                self.update_z2()
                self.apply_z2(z2_config)
                self.wait_z2(z2_config)
                self.Context.z2_configs.pop(0)
                self.Context.save()
            self.get_z2_config(force=True)
            self.Context.try_no = 0
            self.set_info("Update Z2 configuration(s) is completed")

        with self.memoize_stage.clean(commit_on_entrance=False, commit_on_wait=False):
            if self.Parameters.clean:
                self.set_info("Clean Z2 configuration(s)")
                while len(self.Context.z2_configs) > 0:
                    z2_config = self.get_z2_config()
                    self.clean_z2(z2_config)
                    self.Context.z2_configs.pop(0)
                    self.Context.save()

    def wait_z2(self, z2_config):
        self.set_info("Waiting z2 group " + z2_config)
        while True:
            status = self.get_z2_status()
            if status.get("updateStatus") == "FINISHED":
                break
            time.sleep(10)

        if status.get("result") == "SUCCESS":
            self.set_info("Success update group " + z2_config)
            return
        failed_workers = status.get("failedWorkers", [])
        msg = ["Update failed on {} worker(s):".format(len(failed_workers))]
        msg += [" " + worker for worker in failed_workers]
        self.set_info("\n".join(msg))

        if self.Context.try_no > self.Parameters.retry_limit:
            error_message = "Update failed, see details at {}/worker_list?configId={}".format(
                self.Z2_BASE_URL,
                z2_config
            )
            self.set_info("Reached retry limit")
            self.set_info(error_message)
            if self.Parameters.send_error_email:
                notify(
                    self,
                    [self.Parameters.send_error_email],
                    "Deploy task failed (config: {})".format(z2_config),
                    error_message
                )
            raise common.errors.TaskError("Z2 update failed")

        self.Context.try_no += 1
        self.Context.save()
        self.set_info("Retry #{}".format(self.Context.try_no))
        raise sdk2.WaitTime(self.Parameters.retry_time)

    def clean_z2(self, z2_config):
        self.set_info("Clean Z2 group " + z2_config)
        request = self.get_z2_base_request()
        request.update(runAptClean=1)
        self.z2_request("update", request)

    def apply_z2(self, z2_config):
        self.set_info("Apply Z2 configuration for group " + z2_config)
        request = self.get_z2_base_request()
        request.update(forceYes=1)
        self.z2_request("update", request)

    def update_z2(self):
        z2_config = self.get_z2_config()
        self.set_info("Update Z2 configuration group " + z2_config)
        request = self.get_z2_base_request()
        request["items"] = json.dumps(self.get_z2_items())
        logging.debug("Update configuration, request %r", self.dump_request(request))
        self.z2_request("editItems", request)

    def get_z2_status(self):
        return self.z2_request("updateStatus", self.get_z2_base_request(), False)

    def z2_request(self, action, request, post=True):
        url = self.Z2_BASE_URL + "/api/v1/" + action
        logging.info("Z2 API action %s request %s", action, self.dump_request(request))
        response = self.http_request(url, request, post=post)

        if not response.get("success", False):
            raise common.errors.TaskError("Z2 api error response %r", response.get("errorMsg"))
        return response.get("response")

    @retry(Exception, infinity_exception=[Z2TemporaryError])
    def http_request(self, url, request, post):
        logging.info("Request %s post %r", url, post)
        try:
            if post:
                response = requests.post(url, data=request)
            else:
                response = requests.get(url, params=request)
        except Exception as e:
            logging.error("Error http request %s %r", url, e)
            raise common.errors.TaskError("Z2 api error request")
        if response.status_code != 200:
            logging.error("Z2 api response code %r %r", response.status_code, response.text)
            search_result = self.UNKNOWN_ITEM_RE.search(response.text)
            if search_result is not None:
                self.set_info("Z2 error: unknown package " + search_result.group(1))
                raise common.errors.TaskError("Z2 unknown package " + search_result.group(1))
            if self.INPROGRESS_ERR in response.text:
                logging.info("Temporary z2 error")
                raise Z2TemporaryError()
            raise common.errors.TaskError("Z2 api http response code {}".format(response.status_code))
        try:
            data = response.json()
        except Exception as e:
            logging.error("Invalid Z2 api response %r %r", e, response)
            raise common.errors.TaskError("Invalid Z2 api response")
        logging.info("Z2 response %r", data)
        return data

    @staticmethod
    def dump_request(data):
        data_copy = data
        if isinstance(data, dict):
            data_copy = data.copy()
            if "apiKey" in data_copy:
                data_copy["apiKey"] = "HIDDEN"
        return json.dumps(data_copy, sort_keys=True, indent=4, separators=(',',  ': '))
