# -*- coding: utf-8 -*-

import logging

from sandbox import common
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.rtmr.RtmrBuildUsertask import RtmrBuildUsertask
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.common import find_taskconfig, LastResource
from sandbox.projects.rtmr.resources import RtmrConfigTool
from sandbox.sdk2.helpers import subprocess as sp


class RtmrDiffUsertaskConfig(sdk2.Task):
    """Show diff usertask config"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 2 * 1024  # 2Gb

    class Parameters(sdk2.Task.Parameters):
        description = "Show diff usertask config"
        kill_timeout = 5 * 60  # 5 Minutes

        config_task = sdk2.parameters.Task("Config Task", required=True, task_type=RtmrBuildUsertask)
        with sdk2.parameters.String("Cluster name", multiline=True, required=True) as cluster_name:
            _first = True
            for _name in RTMR_CLUSTERS:
                if _first:
                    cluster_name.values[_name] = cluster_name.Value(default=True)
                    _first = False
                else:
                    cluster_name.values[_name] = None

        rtmr_config_resource = LastResource(
            "RTMR Config Tool",
            resource_type=RtmrConfigTool,
            required=True
        )

    class Context(sdk2.Task.Context):
        rtmr_config = None

    def update_config_tool(self):
        if self.Context.rtmr_config is not None:
            return
        rtmr_config_resource = self.Parameters.rtmr_config_resource
        self.Context.rtmr_config = str(sdk2.ResourceData(rtmr_config_resource).path)

    def do_rtmr_config(self, args):
        cmd = [self.Context.rtmr_config]
        cmd.extend(args)
        logging.info("Exec rtmr-config with command line %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise common.errors.TaskError("rtmr-config return code: " + str(proc.returncode))

    def on_execute(self):
        self.update_config_tool()

        config_resource = find_taskconfig(self.Parameters.config_task, self.Parameters.cluster_name)
        if config_resource is None:
            raise common.errors.TaskError(
                "Could not find task config resource")

        taskconfig = str(sdk2.ResourceData(config_resource).path)
        self.do_rtmr_config([
            "tasks",
            "-c", self.Parameters.cluster_name,
            "-i", taskconfig,
            "--dry-run"
        ])
