# -*- coding: utf-8 -*-

import json
import logging
import re
import datetime

import requests
import pytz

from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.common import retry


class Duty(object):
    def __init__(self, login):
        self.login = login
        self.phone = ""


class RtmrDutySwitch(sdk2.Task):
    """Switch RTMR Duty"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 10 * 1024  # 10Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Switch RTMR Duty"
        kill_timeout = 5 * 60  # 5 minutes

        phone_number = sdk2.parameters.String(
            "Duty Phone Number",
            default_value="68195",
            required=True,
        )
        use_calendar = sdk2.parameters.Bool("Set next duty from calendar", default_value=True)
        with use_calendar.value[False]:
            duty_login = sdk2.parameters.String("Login name", required=True)

        use_backup = sdk2.parameters.Bool("Backup duty", default_value=False)
        with use_backup.value[True]:
            backup_phone_number = sdk2.parameters.String(
                "Backup Duty Phone Number",
                default_value="73602",
                required=True,
            )
            with use_calendar.value[False]:
                backup_duty_login = sdk2.parameters.String("Login name", required=True)

        dry_run = sdk2.parameters.Bool("Dry run", default_value=False)

        with sdk2.parameters.Group("Auth") as source_block:
            with use_calendar.value[True]:
                calendar_api_url = sdk2.parameters.Url(
                    "Calendar API url",
                    default_value="https://calendar-api.tools.yandex.net/internal/",
                    required=True,
                )
                calendar_layer_id = sdk2.parameters.Integer(
                    "Calendar layer ID",
                    default_value=75396,
                    required=True,
                )
                calendar_tz = sdk2.parameters.String(
                    "Calendar Timezone",
                    default_value="Europe/Moscow",
                    required=True,
                )
                tvm_token_name = sdk2.parameters.String(
                    "Vault secret name with TVM secret token",
                    default_value="rtmr-duty-calendar-token",
                    required=True
                )
                tvm_self_id = sdk2.parameters.Integer(
                    "Self TVM ID",
                    default_value=2019577,
                    required=True,
                )
                tvm_calendar_id = sdk2.parameters.Integer(
                    "Calendar TVM ID",
                    default_value=2011072,
                    required=True,
                )
            staff_token_name = sdk2.parameters.String(
                "Vault secret name with Staff OAuth token",
                default_value="robot-rtmr-duty-staff",
                required=True
            )
            telegraph_token_name = sdk2.parameters.String(
                "Vault secret name with Telegraph OAuth token",
                default_value="robot-rtmr-duty-telegraph",
                required=True
            )

    @retry(Exception)
    def get_staff_info(self, login):
        params = {
            "login": login,
            "_one": "1",
            "_fields": "login,work_phone,name.first.en,name.last.en",
        }
        auth = "OAuth " + sdk2.Vault.data(self.Parameters.staff_token_name)
        url = "https://staff-api.yandex-team.ru/v3/persons"

        logging.info("Staff request params %r", params)
        try:
            response = requests.get(
                url,
                params=params,
                headers=dict(Authorization=auth)
            )
        except Exception as e:
            logging.error("Error http request to staff api %s %r", url, e)
            raise common.errors.TaskError("Error http request to staff api")
        if response.status_code != 200:
            logging.error("Staff api response code is %r %r", response.status_code, response.text)
            raise common.errors.TaskError("Staff api response code is " + str(response.status_code))
        try:
            data = response.json()
        except Exception as e:
            logging.error("Error decode staff api response %r %r", e, response)
            raise common.errors.TaskError("Error decode staff api response")

        logging.info("Staff api response is %r", data)
        return data

    def get_calendar_duty(self):
        event_name = self.get_calendar_event()
        return filter(lambda el: len(el) > 0, [e.strip() for e in event_name.split(':')[-1].split(',')])[:2]

    @retry(Exception)
    def get_phone_forward(self, phone):
        url = "https://telegraph.yandex.net/api/v3/cucm/translation_pattern/pattern/" + phone
        auth = "OAuth " + sdk2.Vault.data(self.Parameters.telegraph_token_name)

        logging.info("Telegraph request url %s", url)
        try:
            response = requests.get(url, headers=dict(Authorization=auth))
        except Exception as e:
            logging.error("Error http request to telegraph api %s %r", url, e)
            raise common.errors.TaskError("Error http request to telegraph api")
        if response.status_code != 200:
            logging.error("Telegraph api response code is %r %r", response.status_code, response.text)
            raise common.errors.TaskError("Telegraph api response code is " + str(response.status_code))
        try:
            data = response.json()
        except Exception as e:
            logging.error("Error decode telegraph api response %r %r", e, response)
            raise common.errors.TaskError("Error decode telegraph api response")
        logging.info("Telegraph api response is %r", data)
        if data.get("error", True):
            logging.error("Telegraph api response error")
        try:
            return data["result"]["calledPartyTransformationMask"]
        except Exception as e:
            logging.error("Can't get calledPartyTransformationMask %r", e)
        return ""

    @retry(Exception)
    def set_phone_forward(self, dest_phone, phone):
        url = "https://telegraph.yandex.net/api/v3/cucm/translation_pattern/pattern/" + phone
        auth = "OAuth " + sdk2.Vault.data(self.Parameters.telegraph_token_name)

        logging.info("Telegraph request url %s", url)

        if self.Parameters.dry_run:
            logging.info("Dry run mode: switch %s to %s", phone, dest_phone)
            return

        try:
            response = requests.put(
                url,
                data=json.dumps(dict(calledPartyTransformationMask=dest_phone)),
                headers=dict(Authorization=auth)
            )
        except Exception as e:
            logging.error("Error http request to telegraph api %s %r", url, e)
            raise common.errors.TaskError("Error http request to telegraph api")
        if response.status_code != 200:
            logging.error("Telegraph api response code is %r %r", response.status_code, response.text)
            raise common.errors.TaskError("Telegraph api response code is " + str(response.status_code))
        try:
            data = response.json()
        except Exception as e:
            logging.error("Error decode telegraph api response %r %r", e, response)
            raise common.errors.TaskError("Error decode telegraph api response")
        logging.info("Telegraph api response is %r", data)
        if data.get("error", True):
            logging.error("Telegraph api response error")
        return

    def set_next_duty(self, next_duty, phone_number):
        self.set_info("Next duty: {0} with phone {1} for {2}".format(next_duty.login, next_duty.phone, phone_number))

        current_forward = self.get_phone_forward(phone_number)
        self.set_info("Current forward for {0}: {1}".format(phone_number, str(current_forward)))

        forward = "55" + str(next_duty.phone)
        self.set_phone_forward(forward, phone_number)

        if not self.Parameters.dry_run:
            self.set_info("Duty switched for " + phone_number)

    @retry(Exception)
    def do_calendar_request(self, action, params, tvm_ticket):
        url = self.Parameters.calendar_api_url + action
        try:
            response = requests.get(url, params=params, headers={'X-Ya-Service-Ticket': tvm_ticket})
        except Exception as e:
            logging.error("Error http request to calendar api %s %r", url, e)
            raise common.errors.TaskError("Error http request to calendar api")
        if response.status_code != 200:
            logging.error("Calendar api response code is %r %r", response.status_code, response.text)
            raise common.errors.TaskError("Calendar api response code is " + str(response.status_code))
        try:
            data = response.json()
        except Exception as e:
            logging.error("Error decode calendar api response %r %r", e, response)
            raise common.errors.TaskError("Error decode calendar api response")
        logging.info("Calendar api response is %r", data)
        try:
            return data["events"][0]["name"].strip()
        except Exception as e:
            logging.error("Can't get calendar event name %r", e)
        return None

    def get_calendar_event(self):
        now = datetime.datetime.now(pytz.timezone(self.Parameters.calendar_tz))
        format_str = "%Y-%d-%mT%H:%M:%S%z"
        params = {
            "from": now.strftime(format_str),
            "to": (now + datetime.timedelta(seconds=120)).strftime(format_str),
            "layerId": self.Parameters.calendar_layer_id,
            "tz": self.Parameters.calendar_tz,
        }
        return self.do_calendar_request("get-events", params, self.get_service_ticket())

    def get_service_ticket(self):
        from ticket_parser2 import ticket_parser2_pymodule as tvm
        tvm_secret = sdk2.Vault.data(self.Parameters.tvm_token_name)
        tvm_settings = tvm.TvmApiClientSettings(
            self_client_id=self.Parameters.tvm_self_id,
            self_secret=tvm_secret,
            dsts=dict(cal=self.Parameters.tvm_calendar_id))
        tvm_client = tvm.TvmClient(tvm_settings)
        service_ticket = tvm_client.get_service_ticket_for("cal")
        if service_ticket is None:
            raise common.errors.TaskError("Unable to get service ticket")
        return service_ticket

    def on_execute(self):
        phones = [self.Parameters.phone_number.strip()]
        if self.Parameters.use_backup:
            phones.append(self.Parameters.backup_phone_number.strip())
        logins = list()
        if self.Parameters.use_calendar:
            logins = self.get_calendar_duty()
        else:
            logins.append(self.Parameters.duty_login.strip())
            if self.Parameters.use_backup:
                logins.append(self.Parameters.backup_duty_login.strip())

        if len(logins) == 0:
            raise common.errors.TaskError("can't get next duty")
        if len(phones) != len(logins):
            raise common.errors.TaskError("phones and duties mismatch " + str(phones) + ", " + str(logins))

        duties = [Duty(login) for login in logins]

        for number in phones:
            if not re.match(r'^[0-9]+$', number):
                raise common.errors.TaskError("Incorrect destination phone number: " + number)

        for duty in duties:
            if not re.match(r'^[a-zA-Z0-9_-]+$', duty.login):
                raise common.errors.TaskError("Incorrect staff login name: " + duty.login)
            staff_info = self.get_staff_info(duty.login)
            duty.phone = staff_info.get("work_phone")
            if duty.phone is None:
                raise common.errors.TaskError("Can't get next duty phone number for " + duty.login)

        for duty, phone in zip(duties, phones):
            self.set_next_duty(duty, phone)
