# -*- coding: utf-8 -*-

import logging

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.RtmrBuildCommonUserdata import RtmrBuildCommonUserdata
from sandbox.projects.rtmr.RtmrBuildGraph import RtmrBuildGraph
from sandbox.projects.rtmr.RtmrGraphDeploy import RtmrGraphDeploy
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS


class RtmrGraphRelease(sdk2.Task):
    """Release RTMR graph"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Release RTMR user components"
        kill_timeout = 4 * 60 * 60

        with sdk2.parameters.CheckGroup("Clusters") as clusters:
            for _name in RTMR_CLUSTERS:
                clusters.values[_name] = _name

        build_userdata = sdk2.parameters.Bool("Build userdata", default_value=False)
        with build_userdata.value[True]:
            with sdk2.parameters.Group("Build userdata parameters"):
                statface_username = sdk2.parameters.String("Statface username")
                statface_password_vault_owner = sdk2.parameters.String(
                    "Vault secret owner with Statface password"
                )
                statface_password_vault = sdk2.parameters.String("Vault secret name with Statface password")

        build_graphs = sdk2.parameters.Bool("Build graphs", default_value=False)
        with build_graphs.value[True]:
            with sdk2.parameters.Group("Build graphs parameters"):
                arcadia_url = sdk2.parameters.ArcadiaUrl(
                    "Arcadia URL",
                    required=True,
                    default_value="arcadia:/arc/trunk/arcadia"
                )
                parallel_limit = sdk2.parameters.Integer(
                    "Parallel build package limit",
                    required=True,
                    default_value=99
                )
                strip_packages = sdk2.parameters.Bool("Strip debug info", default_value=True)
                create_debug_packages = sdk2.parameters.Bool("Create debug info packages", default_value=False)
                build_type = sdk2.parameters.String("Build type", default_value="release", choices=[("release", "release"), ("profile", "profile")])
                graphs = sdk2.parameters.String("Names of graphs for tasks that need to be built")

        with build_userdata.value[False]:
            add_last_userdata_resource = sdk2.parameters.Bool(
                "Add last common userdata resource if not building userdata",
                default_value=False)
            ya_package_name_for_sign = sdk2.parameters.String("Name of the user to sign the package", required=False)

        with sdk2.parameters.Group("Build task settings") as build_settings:
            build_disk_space = sdk2.parameters.Integer("Execution space (Gb)", default_value=60)

        with sdk2.parameters.Group("YF deploy settings") as yf_deploy_settings:
            slots = sdk2.parameters.Integer(
                "Number of slots for YF function",
                default=None)

            cleanup_missing = sdk2.parameters.Bool(
                "Delete tasks not present in the generated config",
                default=False)

            oauth_token_name = sdk2.parameters.String(
                "Vault secret name with rtmr-deploy OAuth token",
                default="rtmr_deploy_usertask_yf_oauth_token",
                required=True)

    class Context(sdk2.Task.Context):
        arcadia_branch = None
        arcadia_revision = None
        build_graph_taskid = None
        build_userdata_taskid = None
        deploy_yf_taskids_per_cluster = dict()
        user_deploy_taskids_per_cluster = dict()

    def get_version_from_url(self, arcadia_url):
        parsed_url = sdk2.svn.Arcadia.parse_url(arcadia_url)
        if parsed_url.branch is None:
            version = "trunk"
        else:
            version = parsed_url.branch
        if parsed_url.revision is not None:
            version += "@" + parsed_url.revision
        return version

    def build_userdata(self):
        self.set_info("Build common userdata")
        build_task = RtmrBuildCommonUserdata(
            self,
            description="Build RTMR userdata",
            build_disk_space=self.Parameters.build_disk_space,
            statface_username=self.Parameters.statface_username,
            statface_password_vault_owner=self.Parameters.statface_password_vault_owner,
            statface_password_vault=self.Parameters.statface_password_vault,
            ya_package_name_for_sign=self.Parameters.ya_package_name_for_sign,
        )
        build_task.save().enqueue()
        self.Context.build_userdata_taskid = build_task.id

    def build_graphs(self):
        self.set_info("Build graph")
        build_graph = RtmrBuildGraph(
            self,
            description="Build RTMR graph " + self.get_version_from_url(self.Parameters.arcadia_url),
            priority=self.Parameters.priority,
            graphs=self.Parameters.graphs,
            parallel_limit=self.Parameters.parallel_limit,
            clusters=self.get_clusters(),
            strip_packages=self.Parameters.strip_packages,
            create_debug_packages=self.Parameters.create_debug_packages,
            build_type=self.Parameters.build_type,
            build_disk_space=self.Parameters.build_disk_space,
            ya_package_name_for_sign=self.Parameters.ya_package_name_for_sign,
            arcadia_url=sdk2.svn.Arcadia.replace(
                self.Parameters.arcadia_url,
                revision=self.Context.arcadia_revision
            ),
        )
        build_graph.save().enqueue()
        self.Context.build_graph_taskid = build_graph.id

    def get_build_tasks(self):
        tasks = list()
        tasks.append(self.Context.build_userdata_taskid)
        tasks.append(self.Context.build_graph_taskid)
        logging.info("Build tasks %r", tasks)
        return filter(lambda task: task is not None, tasks)

    def get_clusters(self):
        clusters = set(self.Parameters.clusters)
        return list(clusters)

    def get_deploy_yf_tasks(self):
        tasks = list()
        tasks += list(self.Context.deploy_yf_taskids_per_cluster.values())
        logging.info("Deploy to YF tasks: %r", tasks)
        return filter(lambda task: task is not None, tasks)

    def make_userdeploy_tasks(self):
        messages = []
        for cluster_name in self.get_clusters():
            params = dict()
            if not self.Parameters.build_userdata:
                params["add_last_userdata_resource"] = self.Parameters.add_last_userdata_resource

            graphs = self.Parameters.graphs.split(",")
            graphs = filter(bool, [graph.strip() for graph in graphs])

            for graph in graphs:
                task = RtmrGraphDeploy(
                    self,
                    description="Deploy usertasks to " + cluster_name,
                    priority=self.Parameters.priority,
                    cluster=cluster_name,
                    graph=graph,
                    build_task=self,
                    slots=self.Parameters.slots,
                    cleanup_missing=self.Parameters.cleanup_missing,
                    oauth_token_name=self.Parameters.oauth_token_name,
                    **params
                )
                task.save()
                self.Context.user_deploy_taskids_per_cluster[cluster_name] = task.id
                self.Context.save()

                messages.append(
                    "{cluster}, {graph}: <a href=\"https://sandbox.yandex-team.ru/task/{taskid}/view\">{taskid}</a>".format(
                        cluster=cluster_name, graph=graph, taskid=task.id))

        if len(messages) > 0:
            self.set_info("User deploy tasks:\n " + "\n".join(messages), do_escape=False)

    def on_execute(self):
        if not (self.Parameters.build_userdata or
                self.Parameters.build_graphs):
            raise common.errors.TaskError("Need to specify build tasks")

        # Determine proper revision
        with self.memoize_stage.determine_revision(commit_on_entrance=False):
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        # Start build tasks
        with self.memoize_stage.build_userdata(commit_on_entrance=False):
            if self.Parameters.build_userdata:
                self.build_userdata()

        with self.memoize_stage.build_graphs(commit_on_entrance=False):
            if self.Parameters.build_graphs:
                self.build_graphs()

        # Wait build tasks
        with self.memoize_stage.wait_builds(commit_on_entrance=False, commit_on_wait=False):
            rtmr_common.wait_tasks(self.get_build_tasks())

        # Create user deploy tasks
        with self.memoize_stage.user_deploy(commit_on_entrance=False):
            self.make_userdeploy_tasks()
