# -*- coding: utf-8 -*-

from sandbox import common
from sandbox import sdk2

from sandbox.projects.rtmr.common import wait_tasks, get_task_hyperlink
from sandbox.projects.rtmr.clusters import RtmrClustersInfo
from sandbox.projects.rtmr.RtmrMultiDeployUsertask import RtmrMultiDeployUsertask
from sandbox.projects.rtmr.RtmrDiffMirrorConfig import RtmrDiffMirrorConfig
from sandbox.projects.rtmr.RtmrDiffSourcesConfig import RtmrDiffSourcesConfig


class RtmrMetaDeployUsertask(sdk2.Task):
    """Meta-deploy task: create a ready to run draft of multi deploy task"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1024  # 1Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Meta-deploy usertasks to YF and rtmr-host"
        kill_timeout = 1 * 60 * 60  # 1 hour

        build_tasks_per_account = sdk2.parameters.Dict(
            "Build tasks per account",
            required=False)

        build_all_task_id = sdk2.parameters.String(
            "Build all accounts task id",
            required=False)

        build_mirrors_config_task_id = sdk2.parameters.String(
            "Build mirrors config task id",
            required=False)

        build_sources_config_task_id = sdk2.parameters.String(
            "Build sources config task id",
            required=False)

        cluster = sdk2.parameters.String(
            "Cluster to deploy usertasks to",
            required=False)

    class Context(sdk2.Task.Context):
        multi_deploy_taskid = None
        mirror_config_diff_tasks = dict()
        sources_config_diff_task = None

    def create_multi_deploy_task(self):
        description = ""

        if self.Parameters.cluster:
            description += "Cluster: {cluster}\n\n".format(
                cluster=self.Parameters.cluster)

        if len(self.Parameters.build_tasks_per_account) > 0:
            description += "Build tasks to deploy per account:\n"

        for account, taskid in self.Parameters.build_tasks_per_account.items():
            description += "    {account}: ".format(account=account)
            description += get_task_hyperlink(taskid)
            description += "\n"

        if len(self.Parameters.build_tasks_per_account) > 0:
            description += "\n"

        if self.Context.mirror_config_diff_tasks:
            description += "Mirror config diff tasks:\n"
            for mirror, task_id in self.Context.mirror_config_diff_tasks.items():
                description += "    {mirror}: ".format(mirror=mirror)
                description += get_task_hyperlink(task_id)
                description += "\n"

            description += "\n"

        if self.Context.sources_config_diff_task:
            description += "Sources config diff task: "
            description += get_task_hyperlink(self.Context.sources_config_diff_task)
            description += "\n\n"

        build_all_task = None
        if self.Parameters.build_all_task_id:
            build_all_task = sdk2.Task[self.Parameters.build_all_task_id]

        if build_all_task is not None:
            description += "Build task for deploy to rtmr-host: "
            description += get_task_hyperlink(build_all_task.id)

        build_mirrors_config_task = None
        if self.Parameters.build_mirrors_config_task_id:
            build_mirrors_config_task = sdk2.Task[self.Parameters.build_mirrors_config_task_id]

        build_sources_config_task = None
        if self.Parameters.build_sources_config_task_id:
            build_sources_config_task = sdk2.Task[self.Parameters.build_sources_config_task_id]

        task = RtmrMultiDeployUsertask(
            self,
            description=description,
            cluster=self.Parameters.cluster,
            build_tasks_per_account=self.Parameters.build_tasks_per_account,
            build_all_task=build_all_task,
            build_mirrors_config_task=build_mirrors_config_task,
            build_sources_config_task=build_sources_config_task)

        task.save()

        self.Context.multi_deploy_taskid = task.id
        self.Context.save()

        info = "Multi-deploy task: " + get_task_hyperlink(task.id)
        if self.Parameters.cluster:
            info += ", target cluster: "
            info += self.Parameters.cluster

        self.set_info(info, do_escape=False)

    def create_mirror_config_diffs(self):
        if not self.Parameters.build_mirrors_config_task_id:
            return

        config_task = sdk2.Task[self.Parameters.build_mirrors_config_task_id]

        mirrors = config_task.Parameters.mirrors
        if not mirrors:
            mirrors = []
            for cluster_name, cluster_info in RtmrClustersInfo().clusters.items():
                if cluster_info is None:
                    continue

                if cluster_info.is_mirror:
                    mirrors.append(cluster_name)

        if not mirrors:
            return

        msg = "Mirror config diff tasks:\n"

        for mirror in mirrors:
            task = RtmrDiffMirrorConfig(
                self,
                description="Diff mirror config for {mirror}".format(mirror=mirror),
                config_task=config_task,
                cluster_name=mirror)

            task.save()
            task.enqueue()

            msg += "    " + mirror + ": " + get_task_hyperlink(task.id) + "\n"

            self.Context.mirror_config_diff_tasks[mirror] = task.id
            self.Context.save()

        self.set_info(msg, do_escape=False)

    def create_sources_config_diff(self):
        if not self.Parameters.build_sources_config_task_id:
            return

        config_task = sdk2.Task[self.Parameters.build_sources_config_task_id]

        clusters = set(config_task.Parameters.clusters)
        if self.Parameters.cluster not in clusters:
            raise common.errors.TaskError(
                "No sources config for cluster {cluster} was built".format(
                    cluster=self.Parameters.cluster))

        task = RtmrDiffSourcesConfig(
            self,
            description="Diff sources config for {cluster}".format(
                cluster=self.Parameters.cluster),
            config_task=config_task,
            cluster_name=self.Parameters.cluster)

        task.save()
        task.enqueue()

        self.Context.sources_config_diff_task = task.id
        self.Context.save()

        self.set_info(
            "Sources config diff: {task}".format(
                task=get_task_hyperlink(task.id)),
            do_escape=False)

    def on_execute(self):
        with self.memoize_stage.create_mirror_config_diffs(commit_on_entrance=False):
            self.create_mirror_config_diffs()

        with self.memoize_stage.create_sources_config_diff(commit_on_entrance=False):
            self.create_sources_config_diff()

        with self.memoize_stage.create_multi_deploy_task(commit_on_entrance=False):
            self.create_multi_deploy_task()

        with self.memoize_stage.wait_mirror_config_diffs(commit_on_entrance=False, commit_on_wait=False):
            wait_tasks(list(self.Context.mirror_config_diff_tasks.values()))

        if self.Context.sources_config_diff_task is not None:
            with self.memoize_stage.wait_sources_config_diff(commit_on_entrance=False, commit_on_wait=False):
                wait_tasks([self.Context.sources_config_diff_task])
