# -*- coding: utf-8 -*-

import logging

from sandbox import common
from sandbox import sdk2

import sandbox.common.types.task as ctt
import sandbox.projects.rtmr.common as rtmr_common

from sandbox.projects.rtmr.accounts import RtmrAccountsInfo
from sandbox.projects.rtmr.clusters import RtmrClustersInfo
from sandbox.projects.rtmr.RtmrApplyMirrorConfig import RtmrApplyMirrorConfig
from sandbox.projects.rtmr.RtmrApplySourcesConfig import RtmrApplySourcesConfig
from sandbox.projects.rtmr.RtmrBuildMirrorsConfig import RtmrBuildMirrorsConfig
from sandbox.projects.rtmr.RtmrBuildSourcesConfig import RtmrBuildSourcesConfig
from sandbox.projects.rtmr.RtmrDeployGraphYF import RtmrDeployGraphYF

MAX_DEPLOY_TASK_FAILURE_COUNT = 3


class RtmrMultiDeployGraph(sdk2.Task):
    """Multi-deploy graph: perform deploys to YF"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Run a collection of usertasks deployment tasks"
        kill_timeout = 6 * 60 * 60  # 6 hours

        build_tasks_per_account = sdk2.parameters.Dict(
            "Build tasks per account",
            required=False)

        build_mirrors_config_task = sdk2.parameters.Task(
            "Build mirrors config task",
            required=False,
            task_type=RtmrBuildMirrorsConfig)

        build_sources_config_task = sdk2.parameters.Task(
            "Build sources config task",
            required=False,
            task_type=RtmrBuildSourcesConfig)

        cluster = sdk2.parameters.String(
            "Cluster to deploy usertasks to",
            required=False)

    class Context(sdk2.Task.Context):
        apply_mirror_config_tasks = list()
        apply_sources_config_task = None
        deploy_yf_failure_counts = dict()
        deploy_yf_tasks = dict()

    def wait_task(self, taskid):
        task = sdk2.Task[taskid]

        if rtmr_common.is_task_completed(task):
            return

        rtmr_common.wait_tasks([taskid])

        if rtmr_common.is_task_failed(task):
            raise common.errors.TaskError("Task {} failed".format(taskid))

    def deploy_account_to_yf(self, account):
        logging.info("deploy account to YF: {account}".format(account=account))

        if account not in self.Parameters.build_tasks_per_account:
            logging.info("account {account} is not in deploy YF tasks per account dict".format(
                account=account))
            return

        build_task = sdk2.Task[self.Parameters.build_tasks_per_account[account]]

        deploy_task = None
        if account in self.Context.deploy_yf_tasks:
            deploy_task = sdk2.Task[self.Context.deploy_yf_tasks[account]]
            logging.info("using deploy to YF task {task} for account {account} "
                         "from context".format(
                             task=deploy_task.id, account=account))
        else:
            params = dict()
            params["deploy_target"] = "account"

            for cluster_name, accounts in RtmrAccountsInfo().accounts.items():
                if cluster_name != self.Parameters.cluster:
                    continue

                for account_info in accounts:
                    if account_info.name != account:
                        continue

                    if account_info.slots:
                        params["slots"] = account_info.slots

            deploy_task = RtmrDeployGraphYF(
                self,
                description="Deploy {account} graph to YF on {cluster} (from multi-deploy task {mdtask})".format(
                    account=account, cluster=self.Parameters.cluster, mdtask=rtmr_common.get_task_hyperlink(self.id)),
                priority=self.Parameters.priority,
                cluster=self.Parameters.cluster,
                graph=account,
                build_task=build_task.id,
                **params)

            deploy_task.save()

            logging.info("created new deploy to YF task {task} for account {account}".format(
                task=deploy_task.id, account=account))

        if deploy_task.status in ctt.Status.Group.DRAFT:
            self.set_info(
                "Deploying {account} to YF, deploy task: {task}".format(
                    account=account,
                    task=rtmr_common.get_task_hyperlink(deploy_task.id)),
                do_escape=False)

            deploy_task.enqueue()
        elif rtmr_common.is_task_failed(deploy_task):
            if account not in self.Context.deploy_yf_failure_counts:
                self.Context.deploy_yf_failure_counts[account] = 0

            self.Context.deploy_yf_failure_counts[account] += 1
            self.Context.save()

            if self.Context.deploy_yf_failure_counts[account] >= MAX_DEPLOY_TASK_FAILURE_COUNT:
                raise common.errors.TaskError(
                    "Deploy to YF task {task} for {account} failed and failure "
                    "count {count} has exceeded the max number of attempts".format(
                        task=rtmr_common.get_task_hyperlink(deploy_task.id),
                        account=account,
                        count=self.Context.deploy_yf_failure_counts[account]))

            self.set_info(
                "Deploy to YF task {task} for {account} is in failed state, "
                "creating a new task instead".format(
                    task=rtmr_common.get_task_hyperlink(deploy_task.id),
                    account=account),
                do_escape=False)

            params = dict()
            for name, value in iter(deploy_task.Parameters):
                params[name] = value

            deploy_task = RtmrDeployGraphYF(
                self,
                **params)

            deploy_task.save()

            self.set_info(
                "Deploying {account} to YF, deploy task: {task}".format(
                    account=account,
                    task=rtmr_common.get_task_hyperlink(deploy_task.id)),
                do_escape=False)

            deploy_task.enqueue()
        elif not rtmr_common.is_task_completed(deploy_task):
            self.set_info(
                "Waiting for already executing deploy to YF task {task} "
                "for account {account}".format(
                    task=rtmr_common.get_task_hyperlink(deploy_task.id),
                    account=account),
                do_escape=False)

        self.Context.deploy_yf_tasks[account] = deploy_task.id
        self.Context.save()

        self.wait_task(deploy_task.id)

    def deploy_accounts_to_yf(self):
        for account, _ in self.Parameters.build_tasks_per_account.items():
            self.deploy_account_to_yf(account)

    def apply_mirrors_configs(self):
        if not self.Parameters.build_mirrors_config_task:
            return

        mirrors = self.Parameters.build_mirrors_config_task.Parameters.mirrors
        if not mirrors:
            mirrors = []
            for cluster_name, cluster_info in RtmrClustersInfo().clusters.items():
                if cluster_info is None:
                    continue

                if cluster_info.is_mirror:
                    mirrors.append(cluster_name)

        if not mirrors:
            return

        msg = "Apply mirror config tasks:\n"
        for mirror in mirrors:
            task = RtmrApplyMirrorConfig(
                self,
                description="Apply tables config for RTMR mirror {mirror}".format(
                    mirror=mirror),
                config_task=self.Parameters.build_mirrors_config_task,
                cluster_name=mirror)

            task.save()
            task.enqueue()

            msg += "    " + mirror + ": " + rtmr_common.get_task_hyperlink(task.id) + "\n"

            self.Context.apply_mirror_config_tasks.append(task.id)
            self.Context.save()

        self.set_info(msg, do_escape=False)

    def apply_sources_config(self):
        if not self.Parameters.build_sources_config_task:
            return

        clusters = set(self.Parameters.build_sources_config_task.Parameters.clusters)
        if self.Parameters.cluster not in clusters:
            raise common.errors.TaskError(
                "No sources config for cluster {cluster} was built".format(
                    cluster=self.Parameters.cluster))

        task = RtmrApplySourcesConfig(
            self,
            description="Apply sources config for {cluster}".format(
                cluster=self.Parameters.cluster),
            config_task=self.Parameters.build_sources_config_task,
            cluster_name=self.Parameters.cluster)

        task.save()
        task.enqueue()

        self.Context.apply_sources_config_task = task.id
        self.Context.save()

        self.set_info(
            "Apply sources config: {task}".format(
                task=rtmr_common.get_task_hyperlink(task.id)),
            do_escape=False)

    def on_execute(self):
        with self.memoize_stage.apply_mirrors_configs(commit_on_entrance=False):
            self.apply_mirrors_configs()

        if self.Context.apply_mirror_config_tasks:
            with self.memoize_stage.wait_apply_mirrors_configs(commit_on_entrance=False, commit_on_wait=False):
                rtmr_common.wait_tasks(self.Context.apply_mirror_config_tasks)

        self.deploy_accounts_to_yf()

        with self.memoize_stage.apply_sources_config(commit_on_entrance=False):
            self.apply_sources_config()

        if self.Context.apply_sources_config_task is not None:
            rtmr_common.wait_tasks([self.Context.apply_sources_config_task])
