# -*- coding: utf-8 -*-

import logging

from sandbox import common
from sandbox import sdk2

import sandbox.common.types.task as ctt
import sandbox.projects.rtmr.common as rtmr_common

from sandbox.projects.rtmr.accounts import RtmrAccountsInfo
from sandbox.projects.rtmr.clusters import RtmrClustersInfo
from sandbox.projects.rtmr.RtmrApplyMirrorConfig import RtmrApplyMirrorConfig
from sandbox.projects.rtmr.RtmrApplySourcesConfig import RtmrApplySourcesConfig
from sandbox.projects.rtmr.RtmrBuildMirrorsConfig import RtmrBuildMirrorsConfig
from sandbox.projects.rtmr.RtmrBuildSourcesConfig import RtmrBuildSourcesConfig
from sandbox.projects.rtmr.RtmrCleanupUsertaskYF import RtmrCleanupUsertaskYF
from sandbox.projects.rtmr.RtmrDeployUsertaskYF import RtmrDeployUsertaskYF

MAX_DEPLOY_TASK_FAILURE_COUNT = 3


class RtmrMultiDeployUsertask(sdk2.Task):
    """Multi-deploy task: perform deploys to YF and rtmr-host"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Run a collection of usertasks deployment tasks"
        kill_timeout = 6 * 60 * 60  # 6 hours

        build_tasks_per_account = sdk2.parameters.Dict(
            "Build tasks per account",
            required=False)

        build_mirrors_config_task = sdk2.parameters.Task(
            "Build mirrors config task",
            required=False,
            task_type=RtmrBuildMirrorsConfig)

        build_sources_config_task = sdk2.parameters.Task(
            "Build sources config task",
            required=False,
            task_type=RtmrBuildSourcesConfig)

        cluster = sdk2.parameters.String(
            "Cluster to deploy usertasks to",
            required=False)

        with sdk2.parameters.RadioGroup("Filtering options") as filter:
            filter.values.none = filter.Value("none", default=True)
            filter.values.include = filter.Value("include")
            filter.values.exclude = filter.Value("exclude")

        with filter.value["include"]:
            task_ids = sdk2.parameters.String(
                "Comma separated list of task ids which packages need to be "
                "deployed (empty for all packages from usertask build task)",
                required=False)

        with filter.value["exclude"]:
            exclude_tasks = sdk2.parameters.String(
                "Comma separated list of task ids which binaries and config "
                "should be excluded from deploy",
                required=False)

        filter_default_packages = sdk2.parameters.Bool(
            "Filter extraneous packages from default function during deploy",
            default=True)

        cleanup_after_deploy_delay = sdk2.parameters.Integer(
            "Delay in minutes before running cleanup after YF deploy",
            default=10)

    class Context(sdk2.Task.Context):
        apply_mirror_config_tasks = list()
        apply_sources_config_task = None
        cleanup_after_deploy_yf_completed_accounts = dict()
        cleanup_yf_tasks = dict()
        cleanup_yf_failure_counts = dict()
        deploy_yf_failure_counts = dict()
        deploy_yf_tasks = dict()
        rollback_yf_tasks = dict()
        waited_after_deploy_yf_accounts = dict()

    def wait_task(self, taskid):
        task = sdk2.Task[taskid]

        if rtmr_common.is_task_completed(task):
            return

        rtmr_common.wait_tasks([taskid])

        if rtmr_common.is_task_failed(task):
            raise common.errors.TaskError("Task {} failed".format(taskid))

    def deploy_account_to_yf(self, account):
        logging.info("deploy account to YF: {account}".format(account=account))

        if account not in self.Parameters.build_tasks_per_account:
            logging.info("account {account} is not in deploy YF tasks per account dict".format(
                account=account))
            return

        build_task = sdk2.Task[self.Parameters.build_tasks_per_account[account]]

        deploy_task = None
        if account in self.Context.deploy_yf_tasks:
            deploy_task = sdk2.Task[self.Context.deploy_yf_tasks[account]]
            logging.info("using deploy to YF task {task} for account {account} "
                         "from context".format(
                             task=deploy_task.id, account=account))
        else:
            params = dict()
            if self.Parameters.filter == "none":
                params["deploy_target"] = "account"
            elif self.Parameters.filter == "include":
                if self.Parameters.task_ids:
                    params["deploy_target"] = "tasks"
                    params["task_ids"] = self.Parameters.task_ids
                    params["update_previous"] = True
                else:
                    params["deploy_target"] = "account"
            elif self.Parameters.filter == "exclude":
                if self.Parameters.exclude_tasks:
                    params["exclude_target"] = "tasks"
                    params["exclude_tasks"] = self.Parameters.exclude_tasks
                    params["update_previous"] = True

            for cluster_name, accounts in RtmrAccountsInfo().accounts.items():
                if cluster_name != self.Parameters.cluster:
                    continue

                for account_info in accounts:
                    if account_info.name != account:
                        continue

                    if account_info.slots:
                        params["slots"] = account_info.slots

            if self.Parameters.filter_default_packages and account == "default":
                other_accounts = set()
                for other_account in self.Parameters.build_tasks_per_account.keys():
                    if other_account == account:
                        continue
                    other_accounts.add(other_account)

                self_deploy_accounts = set()
                for cluster_name, accounts in RtmrAccountsInfo().accounts.items():
                    if cluster_name != self.Parameters.cluster:
                        continue

                    for account_info in accounts:
                        if account_info.self_deploy:
                            self_deploy_accounts.add(account_info.name)

                other_accounts.update(self_deploy_accounts)

                if self.Parameters.filter == "exclude" and self.Parameters.exclude_tasks:
                    exclude_tasks = params["exclude_tasks"].split(",")
                    exclude_tasks = filter(bool, [t.strip() for t in exclude_tasks])
                    for other_account in other_accounts:
                        account_build_task = sdk2.Task[self.Parameters.build_tasks_per_account[other_account]]
                        package_task = account_build_task.find(
                            task_type=sdk2.Task["RTMR_BUILD_USERTASK"]).first()
                        exclude_tasks.extend(
                            rtmr_common.list_tasks_for_account(
                                self,
                                other_account,
                                self.Parameters.cluster,
                                package_task.Parameters.arcadia_url))
                    params["exclude_tasks"] = ",".join(exclude_tasks)
                else:
                    params["exclude_target"] = "accounts"
                    params["exclude_accounts"] = ",".join(other_accounts)

                params["exclude_packages_mode"] = "relaxed"

            deploy_task = RtmrDeployUsertaskYF(
                self,
                description="Deploy {account} usertasks to YF on {cluster} (from multi-deploy task {mdtask})".format(
                    account=account, cluster=self.Parameters.cluster, mdtask=rtmr_common.get_task_hyperlink(self.id)),
                priority=self.Parameters.priority,
                cluster=self.Parameters.cluster,
                account=account,
                build_task=build_task.id,
                cleanup_first=True,
                cleanup_last=False,
                switch=True,
                **params)

            deploy_task.save()

            logging.info("created new deploy to YF task {task} for account {account}".format(
                task=deploy_task.id, account=account))

        if rtmr_common.is_task_completed(deploy_task):
            logging.info("Deploy task {task} for account {account} is already completed".format(
                task=deploy_task.id, account=account))
            self.do_cleanup_after_deploy(account, deploy_task)
            return

        if deploy_task.status in ctt.Status.Group.DRAFT:
            self.set_info(
                "Deploying {account} to YF, deploy task: {task}".format(
                    account=account,
                    task=rtmr_common.get_task_hyperlink(deploy_task.id)),
                do_escape=False)

            deploy_task.enqueue()
        elif rtmr_common.is_task_failed(deploy_task):
            if account not in self.Context.deploy_yf_failure_counts:
                self.Context.deploy_yf_failure_counts[account] = 0

            self.Context.deploy_yf_failure_counts[account] += 1
            self.Context.save()

            if self.Context.deploy_yf_failure_counts[account] >= MAX_DEPLOY_TASK_FAILURE_COUNT:
                raise common.errors.TaskError(
                    "Deploy to YF task {task} for {account} failed and failure "
                    "count {count} has exceeded the max number of attempts".format(
                        task=rtmr_common.get_task_hyperlink(deploy_task.id),
                        account=account,
                        count=self.Context.deploy_yf_failure_counts[account]))

            self.set_info(
                "Deploy to YF task {task} for {account} is in failed state, "
                "creating a new task instead".format(
                    task=rtmr_common.get_task_hyperlink(deploy_task.id),
                    account=account),
                do_escape=False)

            params = dict()
            for name, value in iter(deploy_task.Parameters):
                params[name] = value

            deploy_task = RtmrDeployUsertaskYF(
                self,
                **params)

            deploy_task.save()

            self.set_info(
                "Deploying {account} to YF, deploy task: {task}".format(
                    account=account,
                    task=rtmr_common.get_task_hyperlink(deploy_task.id)),
                do_escape=False)

            deploy_task.enqueue()
        elif not rtmr_common.is_task_completed(deploy_task):
            self.set_info(
                "Waiting for already executing deploy to YF task {task} "
                "for account {account}".format(
                    task=rtmr_common.get_task_hyperlink(deploy_task.id),
                    account=account),
                do_escape=False)

        self.Context.deploy_yf_tasks[account] = deploy_task.id
        self.Context.save()

        self.wait_task(deploy_task.id)

    def do_cleanup_after_deploy(self, account, deploy_task):
        if account in self.Context.cleanup_after_deploy_yf_completed_accounts:
            logging.info(
                "Cleanup after deploy already performed for account {account}".format(
                    account=account))
            return

        if account in self.Context.cleanup_yf_tasks:
            cleanup_task = sdk2.Task[self.Context.cleanup_yf_tasks[account]]
            logging.info(
                "using cleanup from YF task {task} for account {account} from context".format(
                    task=cleanup_task.id, account=account))
        else:
            if account not in self.Context.rollback_yf_tasks:
                rollback_task = deploy_task.find(
                    task_type=sdk2.Task["RTMR_ROLLBACK_USERTASK_YF"]).first()

                if rollback_task:
                    self.set_info(
                        "Deployed account {account}, rollback task: {task}".format(
                            account=account,
                            task=rtmr_common.get_task_hyperlink(rollback_task.id)),
                        do_escape=False)

                    self.Context.rollback_yf_tasks[account] = rollback_task.id
                    self.Context.save()
                else:
                    self.set_info(
                        "Deployed account {account}, no rollback task".format(
                            account=account))

                    self.Context.rollback_yf_tasks[account] = None
                    self.Context.save()

            if account not in self.Context.waited_after_deploy_yf_accounts:
                if self.Parameters.cleanup_after_deploy_delay is not None and \
                        self.Parameters.cleanup_after_deploy_delay > 0:
                    self.Context.waited_after_deploy_yf_accounts[account] = 1
                    self.Context.save()

                    self.set_info(
                        "Waiting for {n} minutes before running cleanup for account {account}".format(
                            n=self.Parameters.cleanup_after_deploy_delay,
                            account=account))

                    raise sdk2.WaitTime(self.Parameters.cleanup_after_deploy_delay * 60)

            # If we got this far, the account has been deployed and if cleanup
            # after deploy delay is non-zero, the cooldown period has already
            # passed. Need to delete the rollback task if one exists and run
            # cleanup task

            if account in self.Context.rollback_yf_tasks:
                rollback_task_id = self.Context.rollback_yf_tasks[account]
                if rollback_task_id is not None:
                    rollback_task = sdk2.Task[rollback_task_id]
                    if rollback_task.status in ctt.Status.Group.DRAFT:
                        rollback_task.delete()

                        self.set_info(
                            "Deleted rollback task {task} for account {account}".format(
                                task=rtmr_common.get_task_hyperlink(rollback_task_id),
                                account=account),
                            do_escape=False)

                        self.Context.rollback_yf_tasks[account] = None
                        self.Context.save()

            cleanup_task = RtmrCleanupUsertaskYF(
                self,
                description="Cleanup {account} usertasks from YF on {cluster} (from multi-deploy task {mdtask})".format(
                    account=account,
                    cluster=self.Parameters.cluster,
                    mdtask=rtmr_common.get_task_hyperlink(self.id)),
                priority=self.Parameters.priority,
                cluster=self.Parameters.cluster,
                account=account)

            cleanup_task.save()

            logging.info("created new cleanup from YF task {task} for account {account}".format(
                task=cleanup_task.id, account=account))

            self.Context.cleanup_yf_tasks[account] = cleanup_task.id
            self.Context.save()

        if rtmr_common.is_task_completed(cleanup_task):
            self.set_info("Completed cleanup after deploy for account {account}".format(
                account=account))
            self.Context.cleanup_after_deploy_yf_completed_accounts[account] = 1
            self.Context.save()
            return

        if cleanup_task.status in ctt.Status.Group.DRAFT:
            self.set_info(
                "Running cleanup from YF for account {account}, task: {task}".format(
                    account=account,
                    task=rtmr_common.get_task_hyperlink(cleanup_task.id)),
                do_escape=False)
            cleanup_task.enqueue()
        elif rtmr_common.is_task_failed(cleanup_task):
            if account not in self.Context.cleanup_yf_failure_counts:
                self.Context.cleanup_yf_failure_counts[account] = 0

            self.Context.cleanup_yf_failure_counts[account] += 1
            self.Context.save()

            if self.Context.cleanup_yf_failure_counts[account] >= MAX_DEPLOY_TASK_FAILURE_COUNT:
                raise common.errors.TaskError(
                    "Cleanup from YF task {task} for {account} failed and failure "
                    "count {count} has exceeded the max number of attempts".format(
                        task=rtmr_common.get_task_hyperlink(cleanup_task.id),
                        account=account,
                        count=self.Context.cleanup_yf_failure_counts[account]))

            self.set_info(
                "Cleanup from YF task {task} for {account} is in failed state, "
                "creating a new task instead".format(
                    task=rtmr_common.get_task_hyperlink(cleanup_task.id),
                    account=account),
                do_escape=False)

            params = dict()
            for name, value in iter(cleanup_task.Parameters):
                params[name] = value

            cleanup_task = RtmrCleanupUsertaskYF(
                self,
                **params)

            cleanup_task.save()

            self.Context.cleanup_yf_tasks[account] = cleanup_task.id
            self.Context.save()

            self.set_info(
                "Running cleanup from YF for account {account}, task: {task}".format(
                    account=account,
                    task=rtmr_common.get_task_hyperlink(cleanup_task.id)),
                do_escape=False)

            cleanup_task.enqueue()
        else:
            self.set_info(
                "Waiting for already executing cleanup from YF task {task} "
                "for account {account}".format(
                    task=rtmr_common.get_task_hyperlink(cleanup_task.id),
                    account=account),
                do_escape=False)

        self.wait_task(cleanup_task.id)

    def deploy_accounts_to_yf(self):
        for account, _ in self.Parameters.build_tasks_per_account.items():
            self.deploy_account_to_yf(account)

    def apply_mirrors_configs(self):
        if not self.Parameters.build_mirrors_config_task:
            return

        mirrors = self.Parameters.build_mirrors_config_task.Parameters.mirrors
        if not mirrors:
            mirrors = []
            for cluster_name, cluster_info in RtmrClustersInfo().clusters.items():
                if cluster_info is None:
                    continue

                if cluster_info.is_mirror:
                    mirrors.append(cluster_name)

        if not mirrors:
            return

        msg = "Apply mirror config tasks:\n"
        for mirror in mirrors:
            task = RtmrApplyMirrorConfig(
                self,
                description="Apply tables config for RTMR mirror {mirror}".format(
                    mirror=mirror),
                config_task=self.Parameters.build_mirrors_config_task,
                cluster_name=mirror)

            task.save()
            task.enqueue()

            msg += "    " + mirror + ": " + rtmr_common.get_task_hyperlink(task.id) + "\n"

            self.Context.apply_mirror_config_tasks.append(task.id)
            self.Context.save()

        self.set_info(msg, do_escape=False)

    def apply_sources_config(self):
        if not self.Parameters.build_sources_config_task:
            return

        clusters = set(self.Parameters.build_sources_config_task.Parameters.clusters)
        if self.Parameters.cluster not in clusters:
            raise common.errors.TaskError(
                "No sources config for cluster {cluster} was built".format(
                    cluster=self.Parameters.cluster))

        task = RtmrApplySourcesConfig(
            self,
            description="Apply sources config for {cluster}".format(
                cluster=self.Parameters.cluster),
            config_task=self.Parameters.build_sources_config_task,
            cluster_name=self.Parameters.cluster)

        task.save()
        task.enqueue()

        self.Context.apply_sources_config_task = task.id
        self.Context.save()

        self.set_info(
            "Apply sources config: {task}".format(
                task=rtmr_common.get_task_hyperlink(task.id)),
            do_escape=False)

    def on_execute(self):
        with self.memoize_stage.apply_mirrors_configs(commit_on_entrance=False):
            self.apply_mirrors_configs()

        if self.Context.apply_mirror_config_tasks:
            with self.memoize_stage.wait_apply_mirrors_configs(commit_on_entrance=False, commit_on_wait=False):
                rtmr_common.wait_tasks(self.Context.apply_mirror_config_tasks)

        self.deploy_accounts_to_yf()

        with self.memoize_stage.apply_sources_config(commit_on_entrance=False):
            self.apply_sources_config()

        if self.Context.apply_sources_config_task is not None:
            rtmr_common.wait_tasks([self.Context.apply_sources_config_task])
