# -*- coding: utf-8 -*-

import logging
import os

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.RtmrBuildCommonUserdata import RtmrBuildCommonUserdata
from sandbox.projects.rtmr.RtmrBuildUsertask import RtmrBuildUsertask
from sandbox.projects.rtmr.RtmrDeploy import RtmrDeploy
from sandbox.projects.rtmr.RtmrUploadToYF import RtmrUploadToYF
from sandbox.projects.rtmr.RtmrUserDeploy import RtmrUserDeploy
from sandbox.projects.rtmr.RtmrDiffUsertaskConfig import RtmrDiffUsertaskConfig
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.resources import RtmrReleaseDeb, RtmrConfigTool
from sandbox.projects.yf.resources import YfCliLinux

_DAEMON_PACKAGE_TYPES = ["server", "storage", "host", "pusher", "proxy"]
_DAEMON_PACKAGE_BASE = "yandex-search-rtmr-"
_DAEMON_PACKAGE_PATH = "rtmapreduce/packages/daemons"


class RtmrRelease(sdk2.Task):
    """Release components of RTMR"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Release components of RTMR"
        kill_timeout = 4 * 60 * 60

        with sdk2.parameters.CheckGroup("Clusters") as clusters:
            for _name in RTMR_CLUSTERS:
                clusters.values[_name] = _name

        with sdk2.parameters.CheckGroup("Auto deploy clusters") as auto_clusters:
            for _name in RTMR_CLUSTERS:
                auto_clusters.values[_name] = _name

        build_daemons = sdk2.parameters.Bool("Build daemons", default_value=False)
        with build_daemons.value[True]:
            with sdk2.parameters.Group("Build daemons parameters"):
                daemons_arcadia_url = sdk2.parameters.ArcadiaUrl(
                    "Arcadia url",
                    default_value="arcadia:/arc/trunk/arcadia",
                    required=True)
                with sdk2.parameters.CheckGroup("Daemons to build") as daemons_packages:
                    for _name in _DAEMON_PACKAGE_TYPES:
                        daemons_packages.values[_name] = _name

        build_userdata = sdk2.parameters.Bool("Build userdata", default_value=False)
        with build_userdata.value[True]:
            with sdk2.parameters.Group("Build userdata parameters"):
                statface_username = sdk2.parameters.String("Statface username")
                statface_password_vault_owner = sdk2.parameters.String(
                    "Vault secret owner with Statface password"
                )
                statface_password_vault = sdk2.parameters.String("Vault secret name with Statface password")

        with build_userdata.value[False]:
            add_last_userdata_resource = sdk2.parameters.Bool(
                "Add last common userdata resource if not building userdata",
                default_value=False)

        build_usertasks = sdk2.parameters.Bool("Build usertasks", default_value=False)
        with build_usertasks.value[True]:
            with sdk2.parameters.Group("Build usertasks parameters"):
                arcadia_url = sdk2.parameters.ArcadiaUrl(
                    "Arcadia URL",
                    required=True,
                    default_value="arcadia:/arc/trunk/arcadia"
                )
                task_ids = sdk2.parameters.String("Task ids to build (task1,task2... empty for all)")
                parallel_limit = sdk2.parameters.Integer(
                    "Parallel build package limit",
                    required=True,
                    default_value=99
                )
                build_sys = sdk2.parameters.Bool("Include system tasks to build", default_value=True)
                strip_packages = sdk2.parameters.Bool("Strip debug info", default_value=True)
                create_debug_packages = sdk2.parameters.Bool("Create debug info packages", default_value=False)

        with sdk2.parameters.Group("Build task settings") as build_settings:
            ya_package_name_for_sign = sdk2.parameters.String("Name of the user to sign the packages", required=False)
            build_disk_space = sdk2.parameters.Integer("Execution space (Gb)", default_value=60)

        with sdk2.parameters.Group("Deploy settings") as z2_settings:
            rtmr_config_resource = rtmr_common.LastResource(
                "RTMR Config Tool",
                resource_type=RtmrConfigTool,
                required=False
            )
            apply_config = sdk2.parameters.Bool("Apply usertask config", default_value=True)
            restart_host = sdk2.parameters.Bool("Restart host after deploy", default_value=True)
            with restart_host.value[True]:
                with sdk2.parameters.Group("Restart host auth settings") as auth_block:
                    host_secret_name = sdk2.parameters.String("Vault secret name with SSH key")
                    host_secret_owner = sdk2.parameters.String("Vault secret owner")
                    host_remote_user = sdk2.parameters.String("Remote username")
            retry_time = sdk2.parameters.Integer("Z2 retry time (seconds)", required=True, default_value=3 * 60)
            retry_limit = sdk2.parameters.Integer("Z2 retry limit", required=True, default_value=5)
            send_error_email = sdk2.parameters.String("Z2 send error email to")

        with sdk2.parameters.Group("YF upload settings") as yf_settings:
            upload_to_yf = sdk2.parameters.Bool("Upload build to YF", required=False, default_value=False)

            yf_client_resource = rtmr_common.LastYfCliResource(
                "YF client",
                required=False
            )

            deploy_to_yf = sdk2.parameters.Bool(
                "Deploy build to YF",
                required=False,
                default_value=True)

            with deploy_to_yf.value[True]:
                cleanup_first = sdk2.parameters.Bool(
                    "Run rtmr-deploy cleanup before deploying the new account",
                    default=True)

                switch = sdk2.parameters.Bool(
                    "Switch execution to new version",
                    default=True)

                slots = sdk2.parameters.Integer(
                    "Number of slots for YF function",
                    default=None)

                cleanup_missing = sdk2.parameters.Bool(
                    "Delete tasks not present in the generated config",
                    default=False)

                update_previous = sdk2.parameters.Bool(
                    "Update artifacts of previous version instead of full release",
                    default=False)

                oauth_token_name = sdk2.parameters.String(
                    "Vault secret name with rtmr-deploy OAuth token",
                    default="rtmr_deploy_usertask_yf_oauth_token",
                    required=True)

                notification_logins = sdk2.parameters.String(
                    "Comma separated list of staff logins for crash notifications (by e-mail)",
                    required=False,
                    default=None)

        task_config_exclude_accounts_per_cluster = sdk2.parameters.Dict(
            "Accounts to exclude from task config building per cluster",
            required=False,
            default={_name: "" for _name in RTMR_CLUSTERS})

    class Context(sdk2.Task.Context):
        build_daemons_taskid = None
        build_usertasks_taskid = None
        build_userdata_taskid = None
        rtmr_config = None
        yf_client = None
        diff_taskids = list()
        auto_deploy_taskids = list()
        upload_taskids = list()
        deploy_taskids_per_cluster = dict()
        deploy_yf_taskids_per_cluster = dict()
        user_deploy_taskids_per_cluster = dict()

    def make_userdeploy_tasks(self):
        messages = []
        for cluster_name in self.get_clusters():
            params = dict()
            if not self.Parameters.build_userdata:
                params["add_last_userdata_resource"] = self.Parameters.add_last_userdata_resource

            task = RtmrUserDeploy(
                self,
                description="Deploy usertasks to " + cluster_name,
                priority=self.Parameters.priority,
                cluster=cluster_name,
                account="default",
                build_task=self,
                rtmr_config_resource=self.Parameters.rtmr_config_resource,
                cleanup_first=self.Parameters.cleanup_first,
                switch=self.Parameters.switch,
                slots=self.Parameters.slots,
                cleanup_missing=self.Parameters.cleanup_missing,
                update_previous=self.Parameters.update_previous,
                oauth_token_name=self.Parameters.oauth_token_name,
                notification_logins=self.Parameters.notification_logins,
                apply_config=self.Parameters.apply_config,
                restart_host=self.Parameters.restart_host,
                host_secret_name=self.Parameters.host_secret_name,
                host_secret_owner=self.Parameters.host_secret_owner,
                host_remote_user=self.Parameters.host_remote_user,
                retry_time=self.Parameters.retry_time,
                retry_limit=self.Parameters.retry_limit,
                send_error_email=self.Parameters.send_error_email,
                **params
            )
            task.save()
            self.Context.user_deploy_taskids_per_cluster[cluster_name] = task.id
            self.Context.save()

            messages.append(
                "{cluster}: <a href=\"https://sandbox.yandex-team.ru/task/{taskid}/view\">{taskid}</a>".format(
                    cluster=cluster_name, taskid=task.id))

        if len(messages) > 0:
            self.set_info("User deploy tasks:\n " + "\n".join(messages), do_escape=False)

    def make_deploy_task(self, cluster_name):
        params = dict()
        if self.Context.build_daemons_taskid is not None:
            params["daemons_task"] = sdk2.Task[self.Context.build_daemons_taskid]

        if self.Context.build_userdata_taskid is not None:
            params["userdata_task"] = sdk2.Task[self.Context.build_userdata_taskid]
        elif not self.Parameters.build_userdata:
            params["add_last_userdata_resource"] = self.Parameters.add_last_userdata_resource

        if self.Context.build_usertasks_taskid is not None:
            params["usertask_task"] = sdk2.Task[self.Context.build_usertasks_taskid]

        task = RtmrDeploy(
            self,
            description="Deploy to " + cluster_name,
            priority=self.Parameters.priority,
            cluster_name=cluster_name,
            rtmr_config_resource=sdk2.Resource[self.Context.rtmr_config],
            apply_config=self.Parameters.apply_config,
            restart_host=self.Parameters.restart_host,
            host_secret_name=self.Parameters.host_secret_name,
            host_secret_owner=self.Parameters.host_secret_owner,
            host_remote_user=self.Parameters.host_remote_user,
            retry_time=self.Parameters.retry_time,
            retry_limit=self.Parameters.retry_limit,
            send_error_email=self.Parameters.send_error_email,
            **params
        )
        task.save()
        self.Context.deploy_taskids_per_cluster[cluster_name] = task.id
        self.Context.save()
        return task

    def auto_deploy(self):
        tasks = dict()
        for cluster_name in self.Parameters.auto_clusters:
            task = self.make_deploy_task(cluster_name)
            task.enqueue()
            tasks[cluster_name] = task.id
            self.Context.auto_deploy_taskids.append(task.id)
            self.Context.save()
        messages = [
            "{cluster} <a href=\"https://sandbox.yandex-team.ru/task/{taskid}/view\">{taskid}</a>".format(
                cluster=cluster_name,
                taskid=taskid
            )
            for (cluster_name, taskid) in tasks.iteritems()
        ]
        if len(messages) > 0:
            self.set_info("Auto deploy tasks:\n " + "\n ".join(messages), do_escape=False)

    def deploy(self):
        clusters = set(self.Parameters.clusters) - set(self.Parameters.auto_clusters)
        tasks = dict()
        for cluster_name in clusters:
            task = self.make_deploy_task(cluster_name)
            tasks[cluster_name] = task.id
        messages = [
            "{cluster} <a href=\"https://sandbox.yandex-team.ru/task/{taskid}/view\">{taskid}</a>".format(
                cluster=cluster_name,
                taskid=taskid
            )
            for (cluster_name, taskid) in tasks.iteritems()
        ]
        if len(messages) > 0:
            self.set_info("Deploy tasks:\n " + "\n ".join(messages), do_escape=False)

    def get_version_from_url(self, arcadia_url):
        parsed_url = sdk2.svn.Arcadia.parse_url(arcadia_url)
        if parsed_url.branch is None:
            version = "trunk"
        else:
            version = parsed_url.branch
        if parsed_url.revision is not None:
            version += "@" + parsed_url.revision
        return version

    def build_daemons(self):
        self.set_info("Build daemons")
        packages = [
            os.path.join(_DAEMON_PACKAGE_PATH, _DAEMON_PACKAGE_BASE + name, "pkg.json")
            for name in self.Parameters.daemons_packages
        ]
        build_task = sdk2.Task["YA_PACKAGE"](
            self,
            description="Build RTMR Daemons {version}\n{packages}".format(
                version=self.get_version_from_url(self.Parameters.daemons_arcadia_url),
                packages="\n".join(self.Parameters.daemons_packages),
            ),
            checkout_arcadia_from_url=self.Parameters.daemons_arcadia_url,
            use_new_format=True,
            packages=";\n".join(packages),
            host_platform="linux",
            package_type="debian",
            publish_package=True,
            publish_to="search",
            checkout_mode="auto",
            resource_type=RtmrReleaseDeb.name,
        )
        build_task.Requirements.disk_space = self.Parameters.build_disk_space
        build_task.save().enqueue()
        self.Context.build_daemons_taskid = build_task.id

    def build_userdata(self):
        self.set_info("Build common userdata")
        build_task = RtmrBuildCommonUserdata(
            self,
            description="Build RTMR userdata",
            build_disk_space=self.Parameters.build_disk_space,
            statface_username=self.Parameters.statface_username,
            statface_password_vault_owner=self.Parameters.statface_password_vault_owner,
            statface_password_vault=self.Parameters.statface_password_vault,
            ya_package_name_for_sign=self.Parameters.ya_package_name_for_sign,
        )
        build_task.save().enqueue()
        self.Context.build_userdata_taskid = build_task.id

    def build_usertasks(self):
        self.set_info("Build usertasks and config")
        params = dict()
        task_ids = []
        if self.Parameters.task_ids:
            task_ids.append(self.Parameters.task_ids)
        if len(task_ids) > 0:
            params["task_ids"] = ",".join(task_ids)

        params["build_sys"] = self.Parameters.build_sys

        build_task = RtmrBuildUsertask(
            self,
            description="Build RTMR usertasks " + self.get_version_from_url(self.Parameters.arcadia_url),
            priority=self.Parameters.priority,
            arcadia_url=self.Parameters.arcadia_url,
            parallel_limit=self.Parameters.parallel_limit,
            clusters=self.get_clusters_sum(),
            strip_packages=self.Parameters.strip_packages,
            create_debug_packages=self.Parameters.create_debug_packages,
            build_disk_space=self.Parameters.build_disk_space,
            ya_package_name_for_sign=self.Parameters.ya_package_name_for_sign,
            task_config_exclude_accounts_per_cluster=self.Parameters.task_config_exclude_accounts_per_cluster,
            **params
        )
        build_task.save().enqueue()
        self.Context.build_usertasks_taskid = build_task.id

    def get_build_tasks(self):
        tasks = list()
        tasks.append(self.Context.build_daemons_taskid)
        tasks.append(self.Context.build_userdata_taskid)
        tasks.append(self.Context.build_usertasks_taskid)
        logging.info("Build tasks %r", tasks)
        return filter(lambda task: task is not None, tasks)

    def get_upload_tasks(self):
        tasks = list()
        tasks += self.Context.upload_taskids
        logging.info("Upload tasks %r", tasks)
        return filter(lambda task: task is not None, tasks)

    def get_deploy_yf_tasks(self):
        tasks = list()
        tasks += list(self.Context.deploy_yf_taskids_per_cluster.values())
        logging.info("Deploy to YF tasks: %r", tasks)
        return filter(lambda task: task is not None, tasks)

    def get_clusters(self):
        clusters = set(self.Parameters.clusters)
        return list(clusters)

    def get_clusters_sum(self):
        clusters = set(self.Parameters.clusters)
        clusters.update(set(self.Parameters.auto_clusters))
        return list(clusters)

    def start_diff(self):
        tasks = dict()
        for cluster_name in self.get_clusters_sum():
            task = RtmrDiffUsertaskConfig(
                self,
                description="Config diff for cluster " + cluster_name,
                priority=self.Parameters.priority,
                cluster_name=cluster_name,
                config_task=sdk2.Task[self.Context.build_usertasks_taskid],
                rtmr_config_resource=sdk2.Resource[self.Context.rtmr_config],
            )
            task.save().enqueue()
            self.Context.diff_taskids.append(task.id)
            self.Context.save()
            tasks[cluster_name] = task.id
        messages = [
            "{cluster} <a href=\"https://sandbox.yandex-team.ru/task/{taskid}/view\">{taskid}</a>".format(
                cluster=cluster_name,
                taskid=taskid
            )
            for (cluster_name, taskid) in tasks.iteritems()
        ]
        if len(messages) > 0:
            self.set_info("Diff tasks:\n " + "\n ".join(messages), do_escape=False)

    def upload_to_yf(self):
        tasks = dict()

        for cluster_name in self.get_clusters_sum():
            task = RtmrUploadToYF(
                self,
                description="Upload to YF for " + cluster_name,
                priority=self.Parameters.priority,
                cluster_name=cluster_name,
                rtmr_release_task=self.id,
                yf_client_resource=sdk2.Resource[self.Context.yf_client],
            )
            task.save().enqueue()
            self.Context.upload_taskids.append(task.id)
            self.Context.save()
            tasks[cluster_name] = task.id

        messages = [
            "{cluster} <a href=\"https://sandbox.yandex-team.ru/task/{taskid}/view\">{taskid}</a>".format(
                cluster=cluster_name,
                taskid=taskid
            )
            for (cluster_name, taskid) in tasks.iteritems()
        ]
        if len(messages) > 0:
            self.set_info("Upload tasks:\n " + "\n ".join(messages), do_escape=False)

    def on_execute(self):
        if not (self.Parameters.build_daemons or
                self.Parameters.build_userdata or
                self.Parameters.build_usertasks):
            raise common.errors.TaskError("Need to specify build tasks")

        with self.memoize_stage.update_config_tool_resource(commit_on_entrance=False):
            if self.Parameters.rtmr_config_resource is not None:
                self.Context.rtmr_config = self.Parameters.rtmr_config_resource.id
            else:
                self.Context.rtmr_config = sdk2.Resource.find(resource_type=RtmrConfigTool)\
                    .order(-sdk2.Resource.id)\
                    .first().id

        # Start build tasks
        with self.memoize_stage.build_daemons(commit_on_entrance=False):
            if self.Parameters.build_daemons:
                self.build_daemons()

        with self.memoize_stage.build_userdata(commit_on_entrance=False):
            if self.Parameters.build_userdata:
                self.build_userdata()

        with self.memoize_stage.build_usertasks(commit_on_entrance=False):
            if self.Parameters.build_usertasks:
                self.build_usertasks()

        # Wait build tasks
        with self.memoize_stage.wait_builds(commit_on_entrance=False, commit_on_wait=False):
            rtmr_common.wait_tasks(self.get_build_tasks())

        # Start YF upload tasks
        if self.Parameters.upload_to_yf:
            if self.Parameters.yf_client_resource is not None:
                self.Context.yf_client = self.Parameters.yf_client_resource.id
            else:
                self.Context.yf_client = sdk2.Resource.find(
                    resource_type=YfCliLinux,
                    attrs={"released": "stable"}).order(-sdk2.Resource.id).first().id

            with self.memoize_stage.upload_to_yf(commit_on_entrance=False):
                self.upload_to_yf()

            # Wait YF upload tasks
            with self.memoize_stage.wait_uploads(commit_on_entrance=False, commit_on_wait=False):
                rtmr_common.wait_tasks(self.get_upload_tasks())

        # Create user deploy tasks
        if self.Parameters.deploy_to_yf:
            with self.memoize_stage.user_deploy(commit_on_entrance=False):
                self.make_userdeploy_tasks()

        # Start diff tasks
        if self.Parameters.build_usertasks:
            with self.memoize_stage.show_config_diff(commit_on_entrance=False):
                self.start_diff()

        # Make deploy tasks
        with self.memoize_stage.deploy(commit_on_entrance=False):
            self.deploy()

        # Start auto deploy tasks
        with self.memoize_stage.auto_deploy(commit_on_entrance=False):
            self.auto_deploy()

        # Wait auto deploy tasks?
        rtmr_common.wait_tasks(self.Context.auto_deploy_taskids, stop_when_fail=False)
