# -*- coding: utf-8 -*-

import logging

from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.common import LastResource
from sandbox.projects.rtmr.resources import RtmrDeployTool
from sandbox.sdk2.helpers import subprocess as sp


class RtmrRollbackUsertaskYF(sdk2.Task):
    """Rollback usertask to its previous version in YF"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = common.types.client.Tag.Group.LINUX
        disk_space = 2 * 1024  # 2Gb

    class Parameters(sdk2.Task.Parameters):
        description = "Rollback usertask to previous version"
        kill_timeout = 30 * 60  # 30 Minutes

        rtmr_deploy_resource = LastResource(
            "RTMR Deploy Tool",
            resource_type=RtmrDeployTool,
            required=True
        )

        cluster = sdk2.parameters.String("Cluster name", required=True)
        account = sdk2.parameters.String("Name of account to rollback", required=True)

        config_backup_resource = sdk2.parameters.Resource("Resource containing config backup", required=True)
        rollback_version = sdk2.parameters.Integer("Particular version to roll back to", default=None)

        dry_run = sdk2.parameters.Bool("Do not rollback, only show config diff", default=False)

        oauth_token_name = sdk2.parameters.String(
            "Vault secret name with rtmr-deploy OAuth token",
            default="rtmr_deploy_usertask_yf_oauth_token",
            required=True)

    class Context(sdk2.Task.Context):
        rtmr_deploy = None

    def fetch_auth_token(self):
        self.oauth_token = sdk2.Vault.data(self.Parameters.oauth_token_name)

    def update_deploy_tool(self):
        if self.Context.rtmr_deploy is not None:
            return
        rtmr_deploy_resource = self.Parameters.rtmr_deploy_resource
        self.Context.rtmr_deploy = str(sdk2.ResourceData(rtmr_deploy_resource).path)

    def do_rollback(self):
        config_backup = str(sdk2.ResourceData(self.Parameters.config_backup_resource).path)

        cmd = [self.Context.rtmr_deploy,
               "-v",
               "rollback",
               "--cluster", self.Parameters.cluster,
               "--account", self.Parameters.account,
               "--task-config", config_backup,
               "--oauth-token", self.oauth_token,
               ]

        if not self.Parameters.dry_run and self.Parameters.cluster in ["rtmr-man4", "rtmr-vla", "rtmr-sas"]:
            cmd.append("--yes-please-destroy-production-cluster")

        if self.Parameters.rollback_version is not None:
            cmd.extend(["--version", str(self.Parameters.rollback_version)])

        if self.Parameters.dry_run:
            cmd.append("--dry-run")

        logging.info("Executing rtmr-deploy with command line: %r", cmd)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise common.errors.TaskError("rtmr-deploy return code: " + str(proc.returncode))

    def on_execute(self):
        self.fetch_auth_token()
        self.update_deploy_tool()
        self.do_rollback()
