# -*- coding: UTF-8 -*-

import logging
import os
import tarfile

from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.juggler import resource_types
from sandbox.projects.rtmr.common import LastResource
from sandbox.sdk2.helpers import subprocess as sp


class RtmrRunJugglerPlaybook(sdk2.Task):
    """Apply juggler playbook."""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 2 * 1024  # 2Gb

    class Parameters(sdk2.Task.Parameters):
        description = "Apply RTMR Juggler-Ansible playbooks"
        kill_timeout = 24 * 60
        execution_space = 100

        playbook_url = sdk2.parameters.ArcadiaUrl(
            "Svn repository url with playbook",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia/rtmapreduce/monitoring/juggler"
        )
        playbook_file = sdk2.parameters.String(
            "Playbook file",
            required=True,
            default_value="rtmr.yml"
        )
        juggler_secret_name = sdk2.parameters.String(
            "Vault secret name with Juggler OAuth token",
            required=True,
            default_value="rtmr-juggler-oauth"
        )
        juggler_ansible = LastResource(
            "Juggler Ansible resource",
            resource_type=resource_types.JUGGLER_ANSIBLE,
            required=False
        )

    def on_execute(self):
        resource = self.Parameters.juggler_ansible
        if resource is None or resource == "0":
            resource = sdk2.Resource.find(resource_type=resource_types.JUGGLER_ANSIBLE)\
                .order(-sdk2.Resource.id)\
                .first()

        resource_data = sdk2.ResourceData(resource)
        venv_path = str(sdk2.Path("juggler_ansible").absolute())
        tarfile.open(str(resource_data.path)).extractall(venv_path)

        python_path = os.path.join(venv_path, "bin", "python")
        ansible_playbook_path = os.path.join(venv_path, "bin", "ansible-playbook")

        playbook_path = sdk2.svn.Arcadia.get_arcadia_src_dir(self.Parameters.playbook_url)
        inventory_path = os.path.join(playbook_path, "hosts")
        with open(inventory_path, "w"):
            pass

        environment = os.environ.copy()
        environment["JUGGLER_OAUTH_TOKEN"] = sdk2.Vault.data(self.Parameters.juggler_secret_name)

        success = False
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("ansible_playbook")) as pl:
            # Sometimes ansible-playbook fails. Try to apply playbook three
            # times
            for attempt in xrange(0, 3):
                proc = sp.Popen(
                    [python_path, ansible_playbook_path, "-i", "hosts", self.Parameters.playbook_file],
                    cwd=playbook_path,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                    env=environment)
                proc.wait()
                if proc.returncode == 0:
                    success = True
                    break
        if not success:
            raise Exception("Failed to apply playbook")
