# -*- coding: utf-8 -*-

import json

from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.rtmr.resources import RtmrConfigTool
from sandbox.projects.rtmr.common import LastResource
from sandbox.sdk2.helpers import subprocess as sp


class RtmrSearchAcceptTablesSwitch(sdk2.Task):
    """RTMR tables switch on search-accept cluster"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 2 * 1024  # 2Gb

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60
        rtmr_config_resource = LastResource(
            "RTMR Config Tool",
            resource_type=RtmrConfigTool,
            required=False
        )
        prepare_prefix = sdk2.parameters.String(
            "Prepare table prefix",
            required=True,
            default_value="PREPARE_"
        )
        table_prefix = sdk2.parameters.String(
            "Table prefix",
            required=True,
            default_value="pairwise_"
        )
        dry_run = sdk2.parameters.Bool(
            "Dry run",
            default_value=False
        )

    class Context(sdk2.Task.Context):
        rtmr_config = None
        config_file = None

    def do_rtmr_config(self, args):
        cmd = [self.Context.rtmr_config, "-c", "rtmr-search-accept"]
        cmd.extend(args)
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        stdout, _ = proc.communicate()
        if len(stdout) > 0:
            self.set_info(stdout)
        if proc.returncode != 0:
            raise Exception("rtmr-config return code: {}\n".format(proc.returncode))

    def config_load(self):
        self.set_info("Load tables config...")
        self.do_rtmr_config(["-p", "Tables", "-o", self.Context.config_file])

    def config_save(self):
        self.set_info("Save tables config...")
        args = ["-p", "Tables", "-i", self.Context.config_file]
        if self.Parameters.dry_run:
            args.append("--dry-run")
        self.do_rtmr_config(args)

    def config_patch(self):
        self.set_info("Rename table names...")
        with open(self.Context.config_file, "r") as fd:
            config = json.load(fd)
        new_config = dict()
        table_prefix = self.Parameters.table_prefix
        prepare_prefix = self.Parameters.prepare_prefix
        for table_name, table_config in config.iteritems():
            # Swap tables with prefix and without
            if table_name.startswith(prepare_prefix + table_prefix):
                table_name = table_name[len(prepare_prefix):]
            elif table_name.startswith(table_prefix):
                table_name = prepare_prefix + table_name
            new_config[table_name] = table_config
        with open(self.Context.config_file, "w") as fd:
            json.dump(new_config, fd)
        self.set_info("New tables config:\n" + json.dumps(new_config, indent=2))

    def on_execute(self):
        if self.Context.rtmr_config is None:
            rtmr_config_resource = self.Parameters.rtmr_config_resource
            if rtmr_config_resource is None or rtmr_config_resource == "0":
                rtmr_config_resource = sdk2.Resource.find(resource_type=RtmrConfigTool).order(-sdk2.Resource.id).first()
            self.Context.rtmr_config = str(sdk2.ResourceData(rtmr_config_resource).path)
        if self.Context.config_file is None:
            self.Context.config_file = str(sdk2.Path("tables.json").absolute())
        self.config_load()
        self.config_patch()
        self.config_save()
        self.set_info("Done")
