# -*- coding: utf-8 -*-

import logging
import os

import sandbox.projects.rtmr.common as rtmr_common
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS
from sandbox.projects.rtmr.resources import RtmrConfigTool, RtmrQuoterConfig


class RtmrUpdateQuoterConfig(sdk2.Task):
    """Update RTMR Quoter Config"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 10 * 1024  # 10Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Update RTMR Quoter Config"
        kill_timeout = 30 * 60  # 30 minutes

        with sdk2.parameters.CheckGroup("Clusters") as clusters:
            for _name in RTMR_CLUSTERS:
                clusters.values[_name] = _name

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia URL",
            required=True,
            default_value="arcadia:/arc/trunk/arcadia"
        )

        rtmr_config_resource = rtmr_common.LastResource(
            "RTMR Config Tool",
            resource_type=RtmrConfigTool,
            required=False
        )
        force = sdk2.parameters.Bool(
            "Yes, please, destroy production cluster",
            default_value=False
        )

    class Context(sdk2.Task.Context):
        rtmr_config_path = None
        rtmr_gencfg_path = None

    def get_cluster_config_path(self, cluster_name):
        return os.path.join(
            rtmr_common.get_rtmr_configs(self, rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url)),
            cluster_name + ".cfg"
        )

    def save_quoter_config(self, cluster_name):
        resource = RtmrQuoterConfig(
            self,
            "RTMR quoter config for cluster {cluster} {version}".format(
                cluster=cluster_name,
                version=rtmr_common.get_version(self)
            ),
            "quoterconfig-{}.json".format(cluster_name),
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
            cluster=cluster_name,
        )
        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())
        cmd = [
            self.Context.rtmr_gencfg_path,
            "-p", "Quoter",
            "-c", self.get_cluster_config_path(cluster_name),
        ]

        logging.info("Generate quoter config for cluster %s", cluster_name)
        logging.info("run gecfg %r", cmd)
        with open(dst, "w+") as fd:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-gencfg")) as pl:
                proc = sdk2.helpers.subprocess.Popen(cmd, stdout=fd, stderr=pl.stdout)
                proc.wait()
        if proc.returncode != 0:
            raise common.errors.TaskError("Gencfg return code: " + str(proc.returncode))
        resource_data.ready()
        return dst

    def update_config(self, cluster_name):
        quoter_config = self.save_quoter_config(cluster_name)
        cmd = [
            self.Context.rtmr_config_path,
            "-p", "Quoter",
            "-c", cluster_name,
            "-i", quoter_config
        ]
        if self.Parameters.force:
            cmd.append("--yes-please-destroy-production-cluster")

        self.set_info("Apply quoter config for cluster " + cluster_name)
        logging.info("run rtmr-config %r", cmd)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("rtmr-config")) as pl:
            proc = sdk2.helpers.subprocess.Popen(cmd, stdout=sdk2.helpers.subprocess.PIPE, stderr=pl.stdout)
            stdout, _ = proc.communicate()
            returncode = proc.returncode
        if len(stdout) > 0:
            self.set_info(stdout)
        if returncode != 0:
            raise common.errors.TaskError("rtmr-config return code: " + str(returncode))

    def update_config_tool(self):
        if self.Parameters.rtmr_config_resource is not None:
            rtmr_config_resource = self.Parameters.rtmr_config_resource
        else:
            rtmr_config_resource = sdk2.Resource.find(resource_type=RtmrConfigTool)\
                .order(-sdk2.Resource.id)\
                .first().id

        self.Context.rtmr_config_path = str(sdk2.ResourceData(rtmr_config_resource).path)

    def on_execute(self):
        with self.memoize_stage.update_version(commit_on_entrance=False):
            rtmr_common.update_version_info(self, self.Parameters.arcadia_url)

        with self.memoize_stage.update_config_tool_resource(commit_on_entrance=False):
            self.update_config_tool()

        with self.memoize_stage.build_gencfg(commit_on_entrance=False):
            rtmr_common.get_rtmr_gencfg(self, rtmr_common.get_arcadia_url(self, self.Parameters.arcadia_url))

        for cluster_name in self.Parameters.clusters:
            self.update_config(cluster_name)
