# -*- coding: utf-8 -*-

import datetime
import logging
import os
import stat
import tarfile
import tempfile

from sandbox import common
from sandbox import sdk2
from sandbox.projects.rtmr.RtmrBuildUsertask import RtmrBuildUsertask
from sandbox.projects.rtmr.clusters import RTMR_CLUSTERS, RtmrClustersInfo
from sandbox.projects.rtmr.common import LastYfCliResource
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.client as ctc
import sandbox.projects.rtmr.resources as rtmr_resources


class RtmrUploadToYF(sdk2.Task):
    """Upload artifact to yf"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 30 * 1024  # 30Gb

    class Parameters(sdk2.Task.Parameters):
        description = "Upload artifacts to YF"
        kill_timeout = 3600  # 1H

        with sdk2.parameters.String("Cluster name", multiline=True, required=True) as cluster_name:
            _first = True
            for _name in RTMR_CLUSTERS:
                if _first:
                    cluster_name.values[_name] = cluster_name.Value(default=True)
                    _first = False
                else:
                    cluster_name.values[_name] = None

        with sdk2.parameters.Group("Components") as components_block:
            rtmr_release_task = sdk2.parameters.Task("RTMR release task", task_type=RtmrBuildUsertask)

        add_last_userdata_resource = sdk2.parameters.Bool("Add last common userdata resource", default_value=False)

        yf_client_resource = LastYfCliResource(
            "YF Client",
            required=True
        )

    class Context(sdk2.Task.Context):
        yf_client = None
        deploy_taskids = list()
        packages = dict()

    def get_cluster_balancer(self):
        return RtmrClustersInfo().clusters[self.Parameters.cluster_name].balancer

    def collect_packages(self, build_task, resource_type=None):
        packages = []
        tasks = [build_task]

        def recurse(tasks):
            subtasks = []
            for task in tasks:
                t = task.find()
                if t is not None:
                    subtasks.extend(t)

            if len(subtasks) > 0:
                subtasks += recurse(subtasks)
            return subtasks

        tasks += recurse(tasks)

        # collect
        for task in tasks:
            for resource in sdk2.Resource.find(task=task).limit(0):
                if resource.type in resource_type:
                    packages.append(resource)

        if self.Parameters.add_last_userdata_resource:
            common_userdata = sdk2.Resource.find(
                resource_type=rtmr_resources.RtmrUserdataDeb,
                state=common.types.resource.State.READY)\
                .order(-sdk2.Resource.id)\
                .first()
            packages.append(common_userdata)

        return packages

    def update_yf_client(self):
        yf_client_resource = self.Parameters.yf_client_resource
        yf_client_path = str(sdk2.ResourceData(yf_client_resource).path)

        # create a symlink called yf-client
        tmpdir = tempfile.gettempdir()
        yf_client_symlink_path = os.path.join(tmpdir, "yf-client")
        os.symlink(yf_client_path, yf_client_symlink_path)

        # ensure the file is marked as executable
        st = os.stat(yf_client_symlink_path)
        os.chmod(yf_client_symlink_path, st.st_mode | stat.S_IEXEC)

        self.Context.yf_client = yf_client_symlink_path

    def packages_info(self, packages):
        info = ""
        for package in packages:
            if hasattr(package, "resource_name"):
                info += package.resource_name
                if hasattr(package, "resource_version"):
                    info += "="
                    info += package.resource_version
            else:
                info += package.id + "(" + package.description + ")"

            info += "\n"

        return info

    def show_packages(self, packages):
        self.set_info("Packages:\n" + self.packages_info(packages))

    def do_upload_to_yf(self):
        cmd = [self.Context.yf_client]

        packages = self.collect_packages(
            self.Parameters.rtmr_release_task,
            resource_type=[rtmr_resources.RtmrUsertaskDeb, rtmr_resources.RtmrUserdataDeb])

        self.show_packages(packages)

        files_to_upload = []
        for package in packages:
            path = str(sdk2.ResourceData(package).path)
            if path.endswith("tar.gz"):
                with tarfile.open(path) as tar:
                    members = [x for x in tar.getmembers() if x.name.endswith(".deb")]
                    if len(members) > 0:
                        tar.extractall(members=members)
                        files_to_upload += [x.name for x in members]
            else:
                files_to_upload += path

        for attempt in range(0, 5):
            cmd = [self.Context.yf_client,
                   "upload",
                   "-o", "-",
                   "-s", self.get_cluster_balancer()]

            for f in files_to_upload:
                cmd += ["-a", f]

            logging.info("Exec yf client with command line %r", cmd)
            proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.PIPE, bufsize=1)

            # Will poll the running process for stderr and periodically dump it
            # to log

            # Lists containing all lines of stdout/stderr - for use after
            # the process is finished
            stdout_lines = []
            stderr_lines = []

            # Time of the last stderr dump to log
            last_output_dump_time = None

            # Buffer to accumulate stderr lines before dumping them to log
            stderr_buffer_lines = []

            rc = None
            while True:
                stdout_line = proc.stdout.readline()
                stderr_line = proc.stderr.readline()

                if stdout_line:
                    stdout_line = stdout_line.strip()

                if stderr_line:
                    stderr_line = stderr_line.strip()

                if not stdout_line and not stderr_line:
                    rc = proc.poll()
                    if rc is not None:
                        # the process is finished
                        break

                    continue

                stderr_buffer_lines.append(stderr_line)

                stdout_lines.append(stdout_line)
                stderr_lines.append(stderr_line)

                # Dump accumulated lines to log if either:
                # 1. Some new lines were accumulated during the last 5 minutes
                # 2. More than 20 lines of stderr were accumulated

                now = datetime.datetime.now()
                if len(stderr_buffer_lines) >= 20 or \
                        last_output_dump_time is None or \
                        (now - last_output_dump_time).total_seconds() >= 300:

                    if len(stderr_buffer_lines) > 0:
                        logging.info("\n".join(stderr_buffer_lines))

                    stderr_buffer_lines = []
                    last_output_dump_time = now

            if len(stderr_buffer_lines) > 0:
                logging.info("\n".join(stderr_buffer_lines))

            stdout = ""
            if len(stdout_lines) > 0:
                for line in stdout_lines:
                    if not line:
                        continue

                    stdout += line
                    stdout += "\n"

                stdout = stdout.rstrip()
                if len(stdout) > 0:
                    self.set_info(stdout)
                    self.save_yf_artifacts(stdout)

            if rc != 0:
                stderr = ""
                if len(stderr_lines) > 0:
                    for line in stderr_lines:
                        if not line:
                            continue

                        stderr += line
                        stderr += "\n"

                    stderr = stderr.rstrip()

                logging.error(
                    "yf client return code: " + str(rc) + ", stderr: " + stderr)

                raise common.errors.TaskError(
                    "yf client return code: " + str(rc) + ", stderr: " + stderr)
            else:
                return

    def save_yf_artifacts(self, artifacts):
        cluster_name = self.Parameters.cluster_name

        resource = rtmr_resources.RtmrYFArtifacts(
            self,
            "Uploaded aritfacts for YF at {cluster}".format(
                cluster=cluster_name,
            ),
            "yf-artifacts-{}.json".format(cluster_name),
            branch=self.Context.arcadia_branch,
            revision=self.Context.arcadia_revision,
            cluster=cluster_name,
        )

        resource_data = sdk2.ResourceData(resource)
        with open(str(resource_data.path.absolute()), "w+") as fd:
            fd.write(artifacts)
        resource_data.ready()

    def on_execute(self):
        self.update_yf_client()
        self.do_upload_to_yf()
