# -*- coding: utf-8 -*-

from sandbox import common, sdk2

import sandbox.projects.rtmr.resources as rtmr_resources
import sandbox.projects.rtmr.common as rtmr_common

from sandbox.projects.rtmr.RtmrApplyUsertaskConfig import RtmrApplyUsertaskConfig
from sandbox.projects.rtmr.RtmrDeployUsertaskYF import RtmrDeployUsertaskYF


class RtmrUserDeploy(sdk2.Task):
    """Deploy usertasks to YF"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1024  # 1Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Deploy usertasks to YFhost"
        kill_timeout = 3 * 60 * 60  # 3 hours

        cluster = sdk2.parameters.String("Cluster name", required=True)

        build_task = sdk2.parameters.Task(
            "RTMR_USER_RELEASE task used to build artifacts",
            required=True)

        account = sdk2.parameters.String(
            "Name of account to deploy",
            required=True)

        with sdk2.parameters.RadioGroup("Deploy target") as deploy_target:
            deploy_target.values.account = deploy_target.Value("account", default=True)
            deploy_target.values.tasks = deploy_target.Value("tasks")

        with deploy_target.value["account"]:
            with sdk2.parameters.RadioGroup("Exclude target") as exclude_target:
                exclude_target.values.accounts = exclude_target.Value("accounts", default=True)
                exclude_target.values.tasks = exclude_target.Value("tasks")

            with exclude_target.value["accounts"]:
                exclude_accounts = sdk2.parameters.String(
                    "Comma separated list of accounts which tasks' binaries and "
                    "config should be excluded from deploy",
                    required=False)

            with exclude_target.value["tasks"]:
                exclude_tasks = sdk2.parameters.String(
                    "Comma separated list of task ids which binaries and config "
                    "should be excluded from deploy",
                    required=False)

            with sdk2.parameters.RadioGroup("Exclude packages mode") as exclude_packages_mode:
                exclude_packages_mode.values.strict = exclude_packages_mode.Value("strict", default=True)
                exclude_packages_mode.values.relaxed = exclude_packages_mode.Value("relaxed")

        with deploy_target.value["tasks"]:
            task_ids = sdk2.parameters.String(
                "Comma separated list of task ids to deploy (task1,task2,...)",
                required=False)

        config_only = sdk2.parameters.Bool(
            "Deploy only config",
            default=False)

        rtmr_config_resource = rtmr_common.LastResource(
            "RTMR Config Tool",
            resource_type=rtmr_resources.RtmrConfigTool,
            required=True
        )

        with config_only.value[False]:
            with sdk2.parameters.Group("Deploy to YF settings") as yf_settings:
                rtmr_deploy_resource = rtmr_common.LastResource(
                    "RTMR Deploy Tool",
                    resource_type=rtmr_resources.RtmrDeployTool,
                    required=True
                )

                cleanup_first = sdk2.parameters.Bool(
                    "Run rtmr-deploy cleanup before deploying the new version",
                    default=True)

                cleanup_last = sdk2.parameters.Bool(
                    "Run rtmr-deploy cleanup after deploying the new version",
                    default=False)

                switch = sdk2.parameters.Bool(
                    "Switch execution to new version",
                    default=True)

                slots = sdk2.parameters.Integer(
                    "Number of slots for YF function",
                    default=None)

                oauth_token_name = sdk2.parameters.String(
                    "Vault secret name with rtmr-deploy OAuth token",
                    default="rtmr_deploy_usertask_yf_oauth_token",
                    required=True)

                notification_logins = sdk2.parameters.String(
                    "Comma separated list of staff logins for crash notifications (by e-mail)",
                    required=False,
                    default=None)

                with deploy_target.value["account"]:
                    cleanup_missing = sdk2.parameters.Bool(
                        "Delete tasks not present in the generated config",
                        default=False)

                    update_previous = sdk2.parameters.Bool(
                        "Update artifacts of previous version instead of full release",
                        default=False)

    class Context(sdk2.Task.Context):
        deploy_task_ids = list()

    def get_task_ids(self):
        if self.Parameters.deploy_target != "tasks":
            return None

        task_ids = self.Parameters.task_ids.split(",")
        task_ids = filter(bool, [t.strip() for t in task_ids])
        if not task_ids:
            raise common.errors.TaskError(
                "Deploy target was set to \"tasks\" but the actual list of task ids is empty")

        return ",".join(task_ids)

    def get_exclude_task_ids(self):
        if self.Parameters.deploy_target != "account":
            return None

        exclude_tasks = []
        if self.Parameters.exclude_target == "accounts":
            if not self.Parameters.exclude_accounts:
                return None

            exclude_accounts = self.Parameters.exclude_accounts.split(",")
            exclude_accounts = filter(bool, [account.strip() for account in exclude_accounts])

            if not exclude_accounts:
                return None

            package_task = self.Parameters.build_task.find(
                task_type=sdk2.Task["RTMR_BUILD_USERTASK"]).first()

            exclude_tasks = rtmr_common.list_tasks_for_account(
                self,
                ",".join(exclude_accounts),
                self.Parameters.cluster,
                package_task.Parameters.arcadia_url)
        elif self.Parameters.exclude_target == "tasks":
            if not self.Parameters.exclude_tasks:
                return None

            exclude_tasks = self.Parameters.exclude_tasks.split(",")
            exclude_tasks = filter(bool, [t.strip() for t in exclude_tasks])

        if not exclude_tasks:
            return None

        return ",".join(exclude_tasks)

    def apply_config(self):
        self.set_info("Apply usertask config")

        config_task = self.Parameters.build_task.find(
            task_type=sdk2.Task["RTMR_BUILD_USERTASK"]).first()

        if not config_task:
            raise common.erros.TaskError(
                "Failed to find RTMR_BUILD_USERTASK subtask of build task")

        include_tasks = self.get_task_ids()
        exclude_tasks = self.get_exclude_task_ids()

        params = dict()
        if not include_tasks and not exclude_tasks:
            params["filter"] = "none"
        elif include_tasks and exclude_tasks:
            raise common.errors.TaskError(
                "Both include and exclude tasks were specified in parameters")
        elif include_tasks:
            params["filter"] = "include"
            params["task_ids"] = include_tasks
        else:
            params["filter"] = "exclude"
            params["exclude_tasks"] = exclude_tasks

        task = RtmrApplyUsertaskConfig(
            self,
            description="Apply usertasks config on cluster " + self.Parameters.cluster,
            priority=self.Parameters.priority,
            cluster_name=self.Parameters.cluster,
            config_task=config_task,
            rtmr_config_resource=self.Parameters.rtmr_config_resource,
            **params
        )

        task.save().enqueue()

        self.Context.deploy_task_ids.append(task.id)
        self.Context.save()

    def deploy_to_yf(self):
        self.set_info("Deploy usertasks to YF")

        params = dict()
        if self.Parameters.deploy_target == "account":
            params["cleanup_missing"] = self.Parameters.cleanup_missing
            params["update_previous"] = self.Parameters.update_previous
            params["exclude_target"] = self.Parameters.exclude_target
            params["exclude_accounts"] = self.Parameters.exclude_accounts
            params["exclude_tasks"] = self.Parameters.exclude_tasks
            params["exclude_packages_mode"] = self.Parameters.exclude_packages_mode
        elif self.Parameters.deploy_target == "tasks":
            params["task_ids"] = self.get_task_ids()
            params["update_previous"] = True

        task = RtmrDeployUsertaskYF(
            self,
            description="Deploy usertasks to YF on cluster " + self.Parameters.cluster,
            priority=self.Parameters.priority,
            cluster=self.Parameters.cluster,
            account=self.Parameters.account,
            rtmr_deploy_resource=self.Parameters.rtmr_deploy_resource,
            build_task=self.Parameters.build_task.id,
            deploy_target=self.Parameters.deploy_target,
            cleanup_first=self.Parameters.cleanup_first,
            cleanup_last=self.Parameters.cleanup_last,
            switch=self.Parameters.switch,
            slots=self.Parameters.slots,
            oauth_token_name=self.Parameters.oauth_token_name,
            notification_logins=self.Parameters.notification_logins,
            **params
        )

        task.save().enqueue()

        self.Context.deploy_task_ids.append(task.id)
        self.Context.save()

    def on_execute(self):
        if self.Parameters.config_only:
            with self.memoize_stage.apply_config(commit_on_entrance=False):
                self.apply_config()
            with self.memoize_stage.wait_apply_config(commit_on_entrance=False, commit_on_wait=False):
                rtmr_common.wait_tasks(self.Context.deploy_task_ids)
            return

        with self.memoize_stage.deploy_to_yf(commit_on_entrance=False):
            self.deploy_to_yf()

        with self.memoize_stage.wait_deploy_task(commit_on_entrance=False, commit_on_wait=False):
            rtmr_common.wait_tasks(self.Context.deploy_task_ids)
