# -*- coding: UTF-8 -*-

import functools
import json
import logging
import os.path
import os
import shutil
import time

import sandbox.projects.common.arcadia.sdk as arcadiasdk
import sandbox.projects.common.constants as arcadiaconstants
from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn
import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.projects.rtmr.resources import RtmrMirrorConfig, \
    RtmrPackageConfig, RtmrSourceConfig, RtmrUsertaskConfig
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.yf.resources import YfCliLinux

_SCRIPTS_SOURCES = ("rtmapreduce/scripts", "rtmapreduce/config/src")
_RTMR_GENCFG = "rtmapreduce/tools/gencfg/rtmr-gencfg"
_RTMR_CONFIGS = "rtmapreduce/config"


class LastResource(sdk2.parameters.Resource):
    resource_type = None

    @common.utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            state=ctr.State.READY,
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class LastYfCliResource(sdk2.parameters.Resource):
    resource_type=YfCliLinux

    @common.utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs={"released": "stable"},
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


def retry(exception_to_check, infinity_exception=[], tries=5, delay=10, backoff=1.5):
    def deco_retry(f):
        @functools.wraps(f)
        def f_retry(*args, **kwargs):
            mtries, mdelay = tries, delay
            while mtries > 1:
                try:
                    return f(*args, **kwargs)
                except exception_to_check as e:
                    decrement_tries = 1
                    for cls in infinity_exception:
                        if isinstance(e, cls):
                            decrement_tries = 0
                            break
                    logging.info("%r, retry in %r seconds...", e, mdelay)
                    time.sleep(mdelay)
                    mtries -= decrement_tries
                    mdelay *= backoff
            return f(*args, **kwargs)
        return f_retry
    return deco_retry


def notify(task, recipients, subject, message):
    variables = dict(
        task_id=task.id,
        descr=task.descr,
        msg=message,
    )
    subject = subject + " ({task_id})".format(**variables)
    body = (
        u"<html><body><ul>"
        u"<li>Sandbox task: {task_id}</li>"
        u"<li>Sandbox url: <a href=https://sandbox.yandex-team.ru/task/{task_id}/view>"
        u"https://sandbox.yandex-team.ru/task/{task_id}/view</a></li>"
        u"<li>Task description: {descr}</li>"
        u"<li>{msg}</li>"
        u"</ul></body></html>"
    ).format(**variables)
    task.server.notification(
        subject=subject,
        body=body,
        transport=ctn.Transport.EMAIL,
        recipients=recipients,
        charset=ctn.Charset.UTF,
        type=ctn.Type.HTML,
    )


@retry(Exception)
def arcadia_checkout(*args, **kwargs):
    return sdk2.svn.Arcadia.checkout(*args, **kwargs)


def get_rtmr_scripts(task, arcadia_url):
    if task.Context.rtmr_scripts_path != ctm.NotExists:
        return task.Context.rtmr_scripts_path

    task.set_info("Checkout RTMR scripts")
    parsed_url = sdk2.svn.Arcadia.parse_url(arcadia_url)
    for source in _SCRIPTS_SOURCES:
        url = sdk2.svn.Arcadia.replace(
            arcadia_url,
            path=os.path.join(parsed_url.path, source)
        )
        dst_path = sdk2.Path(source)
        dst_path.mkdir(parents=True)
        dst = str(dst_path.absolute())
        arcadia_checkout(url, dst, sdk2.svn.Arcadia.Depth.IMMEDIATES)
        if task.Context.rtmr_scripts_path == ctm.NotExists:
            task.Context.rtmr_scripts_path = dst
    if task.Context.rtmr_scripts_path == ctm.NotExists:
        raise common.errors.TaskError("Can't checkout rmtr scripts")
    return task.Context.rtmr_scripts_path


def is_task_failed(task):
    return task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE


def is_task_completed(task):
    return task.status in ctt.Status.Group.SUCCEED


def get_tasks_by_id(task_ids):
    tasks = list()
    for task_id in task_ids:
        task = sdk2.Task[task_id]
        tasks.append(task.reload())
    return tasks


def stop_tasks(tasks):
    for task in tasks:
        if not is_task_failed(task) and not is_task_completed(task):
            task.stop()


def wait_tasks(task_ids, stop_when_fail=True):
    tasks_to_wait = list()
    logging.info("Wait tasks %r", task_ids)
    tasks = get_tasks_by_id(task_ids)
    for task in tasks:
        logging.info("Task %r status is %r", task.id, task.status)
        if is_task_failed(task):
            if stop_when_fail:
                stop_tasks(tasks)
            raise common.errors.TaskError("Subtask id {} failed".format(task.id))
        if not is_task_completed(task):
            tasks_to_wait.append(task)
    if len(tasks_to_wait) > 0:
        raise sdk2.WaitTask(
            tasks_to_wait,
            list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=False
        )


def update_version_info(task, arcadia_url):
    if task.Context.arcadia_revision != ctm.NotExists and \
       task.Context.arcadia_revision is not None:
        return

    src_path_obj = sdk2.Path("src-info")
    src_path_obj.mkdir()
    src_path = str(src_path_obj.absolute())

    arcadia_checkout(
        arcadia_url,
        src_path,
        sdk2.svn.Arcadia.Depth.IMMEDIATES
    )

    parsed_url = sdk2.svn.Arcadia.parse_url(arcadia_url)
    if parsed_url.branch is not None:
        task.Context.arcadia_branch = parsed_url.branch
    else:
        task.Context.arcadia_branch = "trunk"

    if parsed_url.revision is not None:
        task.Context.arcadia_revision = parsed_url.revision
    else:
        task.Context.arcadia_revision = sdk2.svn.Arcadia.info(src_path)["commit_revision"]
    shutil.rmtree(src_path)


def get_arcadia_url(task, base_arcadia_url):
    if task.Context.arcadia_revision == ctm.NotExists:
        raise common.errors.TaskError("Arcadia revision not found")

    return sdk2.svn.Arcadia.replace(
        base_arcadia_url,
        revision=task.Context.arcadia_revision
    )


def get_rtmr_gencfg(task, arcadia_url):
    if task.Context.rtmr_gencfg_path != ctm.NotExists and \
            task.Context.rtmr_gencfg_path is not None:
        return task.Context.rtmr_gencfg_path

    task.set_info("Checkout RTMR gencfg tool")

    build_path_obj = sdk2.Path("build")
    build_path_obj.mkdir()
    build_path = str(build_path_obj.absolute())
    src_path_obj = sdk2.Path("src")
    src_path_obj.mkdir()
    src_path = str(src_path_obj.absolute())

    arcadia_checkout(
        arcadia_url,
        src_path,
        sdk2.svn.Arcadia.Depth.IMMEDIATES
    )

    tool_path = os.path.dirname(_RTMR_GENCFG)
    to_checkout = []
    path = tool_path
    while len(path) > 0:
        path, _ = os.path.split(path)
        if len(path) > 0:
            to_checkout.append(path)
    to_checkout.reverse()

    for path in to_checkout:
        sdk2.svn.Arcadia.update(
            os.path.join(src_path, path),
            set_depth=sdk2.svn.Arcadia.Depth.IMMEDIATES
        )
    sdk2.svn.Arcadia.update(
        os.path.join(src_path, tool_path),
        set_depth=sdk2.svn.Arcadia.Depth.INFINITY
    )

    task.set_info("Build RTMR gencfg tool")

    cwd = os.path.abspath(os.curdir)
    os.chdir(os.path.join(src_path, tool_path))
    arcadiasdk.do_build(
        arcadiaconstants.YMAKE_BUILD_SYSTEM,
        src_path,
        [tool_path],
        results_dir=build_path,
        clear_build=False,
        checkout=True
    )
    os.chdir(cwd)
    task.Context.rtmr_gencfg_path = os.path.join(build_path, _RTMR_GENCFG)
    shutil.rmtree(src_path)
    return task.Context.rtmr_gencfg_path


def list_all_accounts(task, cluster, arcadia_url=None):
    if arcadia_url is None:
        arcadia_url = get_arcadia_url(task, task.Parameters.arcadia_url)

    rtmr_gencfg_path = get_rtmr_gencfg(task, arcadia_url)

    cluster_config_path = os.path.join(
        get_rtmr_configs(task, arcadia_url),
        cluster + ".cfg"
    )

    cmd = [
        rtmr_gencfg_path,
        "-p", "Accounts",
        "-c", cluster_config_path,
    ]

    logging.info("Executing rtmr-gencfg with command line: %r", cmd)
    proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
    stdout, _ = proc.communicate()
    if proc.returncode != 0:
        raise common.errors.TaskError(
            "rtmr-gencfg error: {retcode} - {stdout}".format(
                retcode=proc.returncode, stdout=stdout))

    return set(json.loads(stdout).keys())


def list_all_tasks(task, cluster, arcadia_url=None):
    if arcadia_url is None:
        arcadia_url = get_arcadia_url(task, task.Parameters.arcadia_url)

    rtmr_gencfg_path = get_rtmr_gencfg(task, arcadia_url)

    cluster_config_path = os.path.join(
        get_rtmr_configs(task, arcadia_url),
        cluster + ".cfg"
    )

    cmd = [
        rtmr_gencfg_path,
        "-p", "TaskList",
        "-c", cluster_config_path,
    ]

    logging.info("Executing rtmr-gencfg with command line: %r", cmd)
    proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
    stdout, _ = proc.communicate()
    if proc.returncode != 0:
        raise common.errors.TaskError(
            "rtmr-gencfg error: {retcode} - {stdout}".format(
                retcode=proc.returncode, stdout=stdout))

    task_ids = set()
    if stdout:
        current_tasks = json.loads(stdout)["TaskIds"]
        for current_task in current_tasks:
            task_ids.add(current_task)

    return task_ids


def list_tasks_for_account(task, account, cluster, arcadia_url=None):
    if arcadia_url is None:
        arcadia_url = get_arcadia_url(task, task.Parameters.arcadia_url)

    rtmr_gencfg_path = get_rtmr_gencfg(task, arcadia_url)

    cluster_config_path = os.path.join(
        get_rtmr_configs(task, arcadia_url),
        cluster + ".cfg"
    )

    cmd = [
        rtmr_gencfg_path,
        "-p", "TasksByAccount",
        "-c", cluster_config_path,
        "-u", account,
    ]

    logging.info("Executing rtmr-gencfg with command line: %r", cmd)
    proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
    stdout, _ = proc.communicate()
    if proc.returncode != 0:
        raise common.errors.TaskError(
            "rtmr-gencfg error: {retcode} - {stdout}".format(
                retcode=proc.returncode, stdout=stdout))

    task_ids = set()
    if stdout:
        current_tasks = json.loads(stdout)["TaskIds"]
        for current_task in current_tasks:
            task_ids.add(current_task)

    return task_ids


def list_packages_for_tasks(task, task_ids, cluster, arcadia_url=None):
    if arcadia_url is None:
        arcadia_url = get_arcadia_url(task, task.Parameters.arcadia_url)

    rtmr_gencfg_path = get_rtmr_gencfg(task, arcadia_url)

    cluster_config_path = os.path.join(
        get_rtmr_configs(task, arcadia_url),
        cluster + ".cfg"
    )

    cmd = [
        rtmr_gencfg_path,
        "-p", "Packages",
        "-c", cluster_config_path,
    ]

    for task_id in task_ids:
        cmd.extend(["-t", task_id])

    logging.info("Executing rtmr-gencfg with command line: %r", cmd)
    proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
    stdout, _ = proc.communicate()
    if proc.returncode != 0:
        raise common.errors.TaskError(
            "rtmr-gencfg error: {retcode} - {stdout}".format(
                retcode=proc.returncode, stdout=stdout))

    return set(json.loads(stdout))


def list_tasks_for_packages(task, packages, cluster, arcadia_url=None):
    if arcadia_url is None:
        arcadia_url = get_arcadia_url(task, task.Parameters.arcadia_url)

    rtmr_gencfg_path = get_rtmr_gencfg(task, arcadia_url)

    cluster_config_path = os.path.join(
        get_rtmr_configs(task, arcadia_url),
        cluster + ".cfg"
    )

    cmd = [
        rtmr_gencfg_path,
        "-p", "TasksByPackage",
        "-c", cluster_config_path,
    ]

    for package in packages:
        cmd.extend(["-g", package])

    logging.info("Executing rtmr-gencfg with command line: %r", cmd)
    proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
    stdout, _ = proc.communicate()
    if proc.returncode != 0:
        raise common.errors.TaskError(
            "rtmr-gencfg error: {retcode} - {stdout}".format(
                retcode=proc.returncode, stdout=stdout))

    return set(json.loads(stdout))


def get_rtmr_configs(task, arcadia_url):
    if task.Context.rtmr_configs_path != ctm.NotExists and \
            task.Context.rtmr_configs_path is not None:
        return task.Context.rtmr_configs_path

    task.set_info("Checkout RTMR configs")

    configs_path_obj = sdk2.Path("configs")
    configs_path_obj.mkdir(parents=True)
    configs_path = str(configs_path_obj.absolute())
    parsed_url = sdk2.svn.Arcadia.parse_url(arcadia_url)

    arcadia_checkout(
        sdk2.svn.Arcadia.replace(
            arcadia_url,
            path=os.path.join(parsed_url.path, _RTMR_CONFIGS)
        ),
        configs_path,
        sdk2.svn.Arcadia.Depth.INFINITY
    )
    task.Context.rtmr_configs_path = configs_path
    return task.Context.rtmr_configs_path


def get_version(task):
    if task.Context.arcadia_revision == ctm.NotExists or \
       task.Context.arcadia_branch == ctm.NotExists:
        raise common.errors.TaskError("Arcadia revision not found")

    return task.Context.arcadia_branch + "@" + task.Context.arcadia_revision


def find_config(build_task, config_type, attrs=dict()):
    tasks = [build_task]
    tasks.extend(build_task.find())
    for task in tasks:
        resource = sdk2.Resource.find(
            resource_type=config_type,
            task=task,
            attrs=attrs
        ).first()

        if resource is not None:
            return resource

    return None


def find_taskconfig(build_task, cluster_name):
    return find_config(build_task, RtmrUsertaskConfig, {"cluster": cluster_name})


def find_packageconfig(build_task):
    return find_config(build_task, RtmrPackageConfig)


def find_mirrorconfig(build_task, cluster_name):
    return find_config(build_task, RtmrMirrorConfig, {"cluster": cluster_name})


def find_sourceconfig(build_task, cluster_name):
    return find_config(build_task, RtmrSourceConfig, {"cluster": cluster_name})


def get_task_hyperlink(task_id):
    return "<a href=\"https://sandbox.yandex-team.ru/task/{task}/view\">{task}</a>".format(
        task=task_id)
