import logging
import tempfile

from sandbox import sdk2
import sandbox.common.types.misc as ctm


class FnsSyncTask(sdk2.Task):

    class Parameters(sdk2.Parameters):
        owner = "zhukovdan"
        description = "Synchronize FNS data copy"
        certificate = sdk2.parameters.YavSecret("Certificate")
        private_key = sdk2.parameters.YavSecret("Private key")
        yt_token = sdk2.parameters.YavSecret("YT token")
        oauth_token = sdk2.parameters.YavSecret("OAuth token")
        force = sdk2.parameters.Bool("Force update", default=False)
        src = sdk2.parameters.String("Source path on FNS server", required=True)
        dest = sdk2.parameters.String("Destination path on YT cluster", required=True)
        yt_cluster = sdk2.parameters.String("YT cluster", default="hahn")
        artifact = sdk2.parameters.String("Reactor artifact path")
        file_regex = sdk2.parameters.String("FNS file name regular expression", default="")

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        from quality.functionality.rtx.counterparty.fns_sync.clients.fns import FnsHttpsClient
        from quality.functionality.rtx.counterparty.fns_sync.clients.yt import YtClient
        from quality.functionality.rtx.counterparty.fns_sync.reactor import InstantiateArtifactsVisitor
        from quality.functionality.rtx.counterparty.fns_sync.transfer import TransferVisitor
        from quality.functionality.rtx.counterparty.fns_sync.tree import build_tree
        from reactor_client import ReactorAPIClientV1
        import yt.wrapper

        certificate = self.Parameters.certificate.data()[self.Parameters.certificate.default_key]
        private_key = self.Parameters.private_key.data()[self.Parameters.private_key.default_key]
        yt_token = self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key]
        oauth_token = self.Parameters.oauth_token.data()[self.Parameters.oauth_token.default_key]

        with tempfile.NamedTemporaryFile(buffering=0) as certfile, tempfile.NamedTemporaryFile(buffering=0) as keyfile:
            certfile.write(certificate.encode("utf-8"))
            keyfile.write(private_key.encode("utf-8"))
            fns_client = FnsHttpsClient(certfile.name, keyfile.name)
        yt_client = YtClient(yt.wrapper.YtClient(proxy=self.Parameters.yt_cluster, token=yt_token))
        reactor_client = ReactorAPIClientV1(base_url="https://reactor.yandex-team.ru", token=oauth_token)

        logging.info("Building source tree")
        src = build_tree(fns_client, root_path=self.Parameters.src, file_regex=self.Parameters.file_regex)
        if self.Parameters.force:
            logging.info("Force update")
            diff = src
        else:
            logging.info("Building destination tree")
            dest = build_tree(yt_client, root_path=self.Parameters.dest)
            logging.info("Calculating diff")
            diff = dest.diff(src)

        if diff is None:
            logging.info("Empty diff")
            return

        logging.info("Starting transfer")
        diff.accept(TransferVisitor(fns_client, yt_client, self.Parameters.src, self.Parameters.dest))
        logging.info("Instantiating artifacts")
        diff.accept(InstantiateArtifactsVisitor(reactor_client, self.Parameters.artifact, self.Parameters.dest))
