import os
import socket

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search.components import component as sc


def _reserve_port():
    # Do not discard returned socket until the service binds to the port,
    # or it will be freed.
    sock = socket.socket(socket.AF_INET6)
    sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    sock.bind(('::', 0))
    port = sock.getsockname()[1]
    return port, sock


class ModelsService(
    sc.WaitUrlComponentMixin,
    sc.ProcessComponentMixinWithShutdownSDK2,
    sc.Component,
):
    """
    Interface for generic models service.
    """

    def __init__(self, name, binary_path, data_path, fresh_path):
        self.name = name
        self._port, self._socket = _reserve_port()
        self._binary_path = binary_path
        self._data_path = data_path
        self._fresh_path = fresh_path

        sc.WaitUrlComponentMixin.__init__(self, url="http://localhost:{}/admin?action=ping", wait_timeout=300, ensure_process=self._ensure_process)
        sc.ProcessComponentMixinWithShutdownSDK2.__init__(
            self,
            args=map(
                str,
                (self._binary_path, "--data", self._data_path, "--realtime", self._fresh_path, "-p", self._port, "--cfg", "begemot.cfg", "--mlock", "yes")
            ),
            shutdown_url="http://localhost:{}/admin?action=shutdown".format(self._port),
            log_prefix=self.name,
        )

    def verify_stderr(self, custom_stderr_path=None):
        """Check stderr output stored in file for errors/warnings existence."""
        stderr_path = custom_stderr_path or self.process.stderr_path
        if not os.path.exists(stderr_path):
            return

        with open(stderr_path) as stderr:
            for line in stderr:
                if "[ERROR]" in line:
                    eh.check_failed("Unexpected stderr log line: '{}'".format(line))
