# -*- coding: utf-8 -*-
import json
import logging
import os

from sandbox import sdk2
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common import task_env
from sandbox.projects.websearch.begemot.common import add_realtime_version_file
from sandbox.sandboxsdk import environments

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace, trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers import subprocess
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import new_resource


WORKING_DIR = "realtime_data"

LM_SERVICE_BASES = {
    "LmServiceBase": "YabsLinearModels",
    "LmServiceCounters": "YabsCounters",
}


@trace_calls
def sky_get(rbtorrent, directory=os.curdir):
    subprocess.check_call(['sky', 'get', '-wud', directory, rbtorrent])


@trace_calls
def prepare_lm_data(yt_client, index_table_path):
    from yt.wrapper import ypath_join

    for table_name, store_dir in LM_SERVICE_BASES.items():
        rule_dir = os.path.join(WORKING_DIR, store_dir)
        if not os.path.exists(rule_dir):
            os.makedirs(rule_dir)

        torrent_id = yt_client.sky_share(ypath_join(index_table_path, table_name))
        index_timestamp = yt_client.get(ypath_join(index_table_path, table_name, '@modification_time'))

        sky_get(rbtorrent=torrent_id, directory=rule_dir)
        index_info = {'create_time': index_timestamp}
        with open(os.path.join(rule_dir, 'index_info'), 'w') as f:
            f.write(json.dumps(index_info, indent=4))


@trace_calls
def prepare_glm_data(yt_client, glm_dumps_path):
    from yt.wrapper import ypath_join

    rule_dir = os.path.join(WORKING_DIR, 'YabsGlm')
    if not os.path.exists(rule_dir):
        os.makedirs(rule_dir)

    try:
        for model_id in yt_client.list(glm_dumps_path):
            with trace('download model', info=dict(model_id=model_id)), open(os.path.join(rule_dir, model_id), 'w') as model_dump:
                model_dump.write(yt_client.read_file(ypath_join(glm_dumps_path, model_id)).read())
    except Exception as e:
        logging.error(e)


@trace_calls
def make_archive(task_id, archive_path, dirs):
    version_file = add_realtime_version_file(WORKING_DIR, task_id)
    subprocess.check_call(['tar', '-C', WORKING_DIR, '-cf', archive_path] + dirs + [version_file])


class BuildYabsRtModelsPackage(sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        """See SEARCH-11128."""
        disk_space = 20 * 1024  # 20 Gb (real usage: ~10 Gb)
        ram = 48 * 1024  # 48 Gb
        environments = [
            environments.PipEnvironment('yandex-yt', version='0.10.8'),
        ]

    class Parameters(sdk2.Parameters):
        begemot_shard = sdk2.parameters.String("begemot_shard", hint=True, default='YabsRtModels')
        yt_token_secret_name = sdk2.parameters.String("yt_token_secret_name", hint=True, default='yt_token_for_testenv')
        yt_proxy = sdk2.parameters.String("YT proxy", default="hahn")
        index_table_path = sdk2.parameters.String("index_table_path", hint=True, default='//home/yabs-cs/key_2/v2/import/')
        glm_dumps_path = sdk2.parameters.String(
            "glm_dumps_path", hint=True, default='//home/yabs-cs/ads_quality/linear_models/service_under_meta_glm'
        )
        rules_to_deploy = sdk2.parameters.List(
            "rules_to_deploy",
            default=list(LM_SERVICE_BASES.values()) + ['YabsGlm']
        )
        release_resource = sdk2.parameters.Bool("Release resource as stable", default=False, do_not_copy=True)
        resource_ttl = sdk2.parameters.Integer("Resource ttl in days", default=2)
        upload_to_mds = sdk2.parameters.Bool("Upload resource to MDS", default=False)
        with upload_to_mds.value[True]:
            sync_upload_to_mds = sdk2.parameters.Bool("Sync upload (wait for MDS upload)", default=False)

        build_resource_type = sdk2.parameters.String("Package resource type", default="YABS_LINEAR_MODELS_DATA")
        with sdk2.parameters.Output:
            package_resource = sdk2.parameters.Resource('Result resource id', required=True)

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        from yt.wrapper import YtClient

        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_secret_name)

        lm_data_yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)
        prepare_lm_data(lm_data_yt_client, self.Parameters.index_table_path)

        glm_data_yt_client = YtClient(proxy="hahn", token=yt_token)
        prepare_glm_data(glm_data_yt_client, self.Parameters.glm_dumps_path)

        archive_path = os.path.abspath('realtime_wizard_data.tar')
        make_archive(self.id, archive_path, self.Parameters.rules_to_deploy)
        resource_attributes = dict(
            shard=self.Parameters.begemot_shard,
            ttl=self.Parameters.resource_ttl,
            version=self.id,
        )
        if self.Parameters.upload_to_mds:
            resource_attributes.update(sync_upload_to_mds=self.Parameters.sync_upload_to_mds)

        resource_class = sdk2.Resource[self.Parameters.build_resource_type]
        resource = new_resource(
            resource_class,
            self,
            resource_class.name + ' for {}'.format(self.Parameters.begemot_shard),
            archive_path,
            **resource_attributes
        )
        if self.Parameters.release_resource:
            resource.released = ReleaseStatus.STABLE

        with self.memoize_stage.set_output:
            self.Parameters.package_resource = resource
