import logging

from sandbox import sdk2
from sandbox.common.types.misc import NotExists
from sandbox.sandboxsdk import environments

from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.runtime_models.components.yabs.BuildYabsRtModelsPackage import BuildYabsRtModelsPackage
from sandbox.projects.samogon.api import SamogonClusterAPI


def find_stable_index_table_path(table_paths):
    cluster_api_template = "https://clusterapi-{}.n.yandex-team.ru"
    best_yabscs_contour = None
    best_yabscs_version = None
    result_table_path = None
    for contour, table_path in table_paths.items():
        try:
            base_url = cluster_api_template.format(contour)
            samogon_api = SamogonClusterAPI(base_url)

            resource_id = samogon_api.get_meta_resource_id()

            meta_resource = sdk2.Resource[resource_id]
            is_released = getattr(meta_resource, "released") == "stable"
            if not is_released:
                logging.info("contour %s is not stable", contour)
                continue

            version = getattr(meta_resource, "resource_version", "0")
            if result_table_path is None:
                best_yabscs_contour = contour
                best_yabscs_version = version
                result_table_path = table_path
            if best_yabscs_version < version:
                best_yabscs_contour = contour
                best_yabscs_version = version
                result_table_path = table_path
        except Exception as e:
            logging.error(e)

    if result_table_path is None:
        logging.error("Can not find a stable yabscs contour")
        return None

    logging.debug(
        "Will use contour=%s, version=%s, index_table_path=%s",
        best_yabscs_contour, best_yabscs_version, result_table_path)

    return result_table_path


def select_best_yt_proxy(yt_proxies, yt_token, index_table_path):
    from yt.wrapper import YtClient, ypath_join

    best_proxy = None
    best_proxy_timestamp = 0
    for proxy in yt_proxies:
        try:
            yt_client = YtClient(proxy=proxy, token=yt_token)
            timestamp = yt_client.get(ypath_join(index_table_path, "LmServiceBase", "@modification_time"))
            if best_proxy_timestamp < timestamp:
                best_proxy = proxy
                best_proxy_timestamp = timestamp
        except Exception as e:
            logging.error(e)

    if best_proxy is None:
        logging.error("All resource_proxy_urls are unreachable")
        return None

    return best_proxy


class BuildYabsRtModelsPackageUsingStableData(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        environments = [
            environments.PipEnvironment("yandex-yt", version="0.10.8"),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BuildYabsRtModelsPackage.Parameters):
        database_proxies = sdk2.parameters.List("database_proxies", default=["hahn", "arnold"])
        index_table_paths = sdk2.parameters.Dict(
            "index_table_paths", default={
                "yabscs1": "//home/yabs-cs/key_1/v2/import/",
                "yabscs2": "//home/yabs-cs/key_2/v2/import/"
            }
        )
        release_resource = BuildYabsRtModelsPackage.Parameters.release_resource(default=True)
        yt_proxy = None
        index_table_path = None
        push_tasks_resource = True
        build_resource_type = BuildYabsRtModelsPackage.Parameters.build_resource_type()

    def on_execute(self):
        if self.Context.child_task_id is NotExists:
            stable_index_table_path = find_stable_index_table_path(self.Parameters.index_table_paths)
            logging.debug("Stable index table path: %s", stable_index_table_path)

            token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_secret_name)
            yt_proxy = select_best_yt_proxy(self.Parameters.database_proxies, token, stable_index_table_path)
            logging.info("Best YT proxy: %s", yt_proxy)

            child_parameters = {
                name: value
                for name, value in self.Parameters
                if not getattr(type(self).Parameters, name).output and hasattr(BuildYabsRtModelsPackage.Parameters, name)
            }
            logging.info('Child task parameters:\n%s', child_parameters)

            child_parameters.update(dict(
                yt_proxy=yt_proxy,
                index_table_path=stable_index_table_path,
            ))

            child_task = BuildYabsRtModelsPackage(
                self,
                owner=self.owner,
                tags=self.Parameters.tags,
                hints=list(self.hints),
                description=self.Parameters.description,
                **child_parameters
            ).enqueue()

            self.Context.child_task_id = child_task.id

        check_tasks(self, self.Context.child_task_id)
