# -*- coding: utf-8 -*-

import logging
import os
import sandbox.sandboxsdk.svn as sdk_svn

from sandbox import sdk2
from sandbox.sandboxsdk import ssh
from sandbox.projects.saas.DeployFerrymanConfig import DeployFerrymanConfig


ARCADIA_CONFIG_PATH = '/yweb/querydata/querydata_indexer_saas/ferryman/static_configs/data'
LOCAL_CONFIG_DIR_TMPL = 'configs/{}'
LOCAL_CONFIG_NAME_TMPL = 'config-{}.pb.txt'


class AddFerrymanConfig(DeployFerrymanConfig):
    """
    Create and add SaaS Ferryman configuration using variables
    """
    class Parameters(sdk2.Parameters):
        ferryman_configs_dir = sdk2.parameters.ArcadiaUrl(
            'Path to ferryman configs directory',
            required=True,
            default_value=sdk_svn.Arcadia.trunk_url() + ARCADIA_CONFIG_PATH
        )
        ferryman_name = sdk2.parameters.String('Ferryman name', required=True)
        ferryman_ctype = sdk2.parameters.String('Ctype', required=True)
        yt_cluster = sdk2.parameters.String('Yt cluster (hahn, arnold)', default='hahn', required=True)
        input_size = sdk2.parameters.String('Doc count (input size)', required=True)
        ferryman_type = sdk2.parameters.String('Ferryman type (ytpull or snapshot)', required=True)
        row_processor = sdk2.parameters.String('Row processor', required=True)
        service_type = sdk2.parameters.String('Service type (search or kv)', required=True)
        dry_run = sdk2.parameters.Bool('Dry run mode', default_value=False)

    def on_prepare(self):
        sdk2.svn.Arcadia.checkout(self.Parameters.ferryman_configs_dir, 'configs')

    def commit(self, message):
        config_dir = LOCAL_CONFIG_DIR_TMPL.format(self.Parameters.ferryman_ctype)
        with ssh.Key(self, 'RTYSERVER-ROBOT', 'ssh_key'):
            return sdk2.svn.Arcadia.commit(config_dir, message, user='saas-robot')

    def find_config_resource(self, task_id):
        task_resources = [r for r in sdk2.resource.Resource.find(task_id=task_id).limit(0)]
        for resource in task_resources:
            if 'FERRYMAN_CONFIG' in resource.type and resource.service == self.Parameters.ferryman_ctype + '/' + self.Parameters.ferryman_name:
                return resource.id

    def svn_add(self, file_path):
        try:
            sdk2.svn.Arcadia.add(file_path)
        except sdk_svn.SvnError:
            logging.error('Path %s already added to svn' % file_path)

    def get_sandbox_task_id(self, testenv_answer, revision=None):
        sandbox_id = ''
        for row in testenv_answer['rows']:
            if ((revision and int(revision) == row['revision']) or
                    (self.Parameters.ferryman_name in row['revision_info/comment'] and self.Parameters.ferryman_ctype in row['revision_info/comment'])):
                if row['task_id']:
                    sandbox_id = row['task_id']
                    break
        if sandbox_id:
            logging.info('Target sandbox id %s', sandbox_id)
        return sandbox_id

    def on_execute(self):
        import saas.tools.ssm.ssm_namespaces as ssm_namespaces
        from saas.tools.ssm.modules.misc import load_and_write_resources
        from saas.tools.ssm.ssm_ferryman import FerrymanNanny

        # Load resources
        load_and_write_resources('/tmpl/ferryman')
        load_and_write_resources('/conf')

        # Prepare clients
        os.environ['NANNY_TOKEN'] = sdk2.Vault.data('RTYSERVER-ROBOT', 'nanny_token')
        ferryman_client = FerrymanNanny(self.Parameters.ferryman_name, self.Parameters.yt_cluster,
                                        self.Parameters.row_processor, self.Parameters.ferryman_ctype,
                                        self.Parameters.input_size,
                                        service_type=self.Parameters.service_type,
                                        ferryman_type=self.Parameters.ferryman_type)
        # Prepare configuration data
        config_data_raw = ferryman_client.render_config()
        config_data = ssm_namespaces.parse_config(config_data_raw)
        config_dir = LOCAL_CONFIG_DIR_TMPL.format(self.Parameters.ferryman_ctype)
        config_name = LOCAL_CONFIG_NAME_TMPL.format(self.Parameters.ferryman_name)
        config_path = os.path.join(config_dir, config_name)

        # Check for existing ctype
        if not os.path.isdir(config_dir):
            logging.warning('Directory %s was not found. Creating' % config_dir)
            os.mkdir(config_dir)
            self.svn_add(config_dir)

        # Write configuration
        ssm_namespaces.write_config(config_path, config_data)

        self.svn_add(config_path)

        self.Context.svn_diff = sdk2.svn.Arcadia.diff('configs')
        self.Context.commit = 'dry_run'
        self.Context.target_task_id = 'dry_run'
        self.Context.target_resource_id = 'dry_run'

        logging.info('Arcadia svn diff\n%s', self.Context.svn_diff)

        if not self.Parameters.dry_run:
            # Commit changes
            logging.info('Committing changes.')
            response = self.commit('SKIP_CHECK  New ferryman config for service {} ctype {}'.format(self.Parameters.ferryman_name,
                                                                                                    self.Parameters.ferryman_ctype))
            rev_id = self.get_revision_id_from_message(response)
            logging.info('Revision: ' + rev_id)

            # Get sandbox task
            subtask_id = self.get_id_from_testenv(rev_id)

            # Get config resource
            resource_id = self.find_config_resource(subtask_id)

            # Update context
            self.Context.commit = rev_id
            self.Context.target_task_id = subtask_id
            self.Context.target_resource_id = resource_id
        else:
            logging.info('Dry run mode was enabled. Changes will not be committed.')
