# -*- coding: utf-8 -*-
import re
import json
import textwrap

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.projects.saas.common.semaphores as semaphores

from sandbox.sdk2.helpers import subprocess
from sandbox.projects.gencfg.BaseGencfgGuiRequest import BaseGencfgGuiRequest


class SaasGroupsInfo(sdk2.Resource):
    """JSON groups description"""


class AllocateInSaas(BaseGencfgGuiRequest):
    """ Allocate groups in SaaS Cloud """
    volume_size_re = re.compile(r'^(?P<size>\d+(\.\d+)?)\s*Gb$', flags=re.IGNORECASE)
    min_volume_sizes = {
        '': 10,
        '/': 10,
        '/db/bsconfig/webstate': 1,
        '/ssd': 50,
        '/logs': 20
    }

    class Requirements(BaseGencfgGuiRequest.Requirements):
        semaphores = semaphores.SaasSemaphore.gencfg

    class Context(BaseGencfgGuiRequest.Context):
        errors = []

    class Parameters(BaseGencfgGuiRequest.Parameters):
        with parameters.Group('Request params') as request_params:
            base_name = parameters.String('Base Name', required=True)
            instance_count = parameters.Integer('Instance count', required=True)
            cpu_requirements = parameters.Integer('Required CPU power in Kimkim power units', required=True)
            mem_requirements = parameters.Integer('Required RAM in Gb', required=True)
            shards_count = parameters.Integer('Max hosts per switch', default=1, required=True)
            group_tags = parameters.JSON('Tags json description, for example: { "ctype":"prod", "itype":"rtyserver" }', required=True)
            no_indexing = parameters.Bool('With saas_no_indexing tag', default=False, required=True)
            owners = parameters.List(label='Owners (override default)', value_type=parameters.String)
            template_group = parameters.String('Custom template group', required=False)
            network_macro = parameters.String('Custom template group', required=False)
            with sdk2.parameters.CheckGroup('Location') as locations:
                locations.values.MAN = locations.Value('Mantsala', checked=True)
                locations.values.SAS = locations.Value('Sasovo', checked=True)
                locations.values.VLA = locations.Value('Vladimir', checked=True)
                locations.values.IVA = locations.Value('Ivanteevka', checked=False)
            volumes = parameters.JSON('Volumes for groups', required=True, default=[
                {'guest_mp': '', 'host_mp_root': '/place', 'quota': '{} Gb'.format(10), 'symlinks': []},
                {'guest_mp': '/', 'host_mp_root': '/place', 'quota': '{} Gb'.format(10), 'symlinks': []},
                {'guest_mp': '/db/bsconfig/webstate', 'host_mp_root': '/place', 'quota': '1.0 Gb', 'symlinks': ['/state']},
                {'guest_mp': '/cores', 'host_mp_root': '/place', 'quota': '{} Gb'.format(1), 'symlinks': []},
                {'guest_mp': '/ssd', 'host_mp_root': '/ssd', 'quota': '{} Gb'.format(50), 'symlinks': ['/data']},
                {'guest_mp': '/logs', 'host_mp_root': '/place', 'quota': '{} Gb'.format(20), 'symlinks': ['/usr/local/www/logs']}])
            io_limits = parameters.JSON('IO limits for groups', required=True, default={
                'bandwidth': {'ssd': {'read': 25, 'write': 20},     'hdd': {'read': 15, 'write': 10}},
                'ops':       {'ssd': {'read': 2000, 'write': 1000}, 'hdd': {'read': 1000, 'write': 500}},
            })

        with parameters.Group('Execution') as execution:
            dry_run = sdk2.parameters.Bool(
                'Dry run',
                description='Run the whole procedure, but do not commit results',
                default=True
            )
            arcadia_revision = sdk2.parameters.String(
                'Base revision',
                description='Allocate, check and commit against this revision',
                required=False
            )
            use_last_resources = sdk2.parameters.Bool(
                'Use last released resources',
                required=True,
                default=False
            )

            commit_message = parameters.String('Commit message', default='', required=False)

    def on_save(self):
        super(AllocateInSaas, self)
        for volume in self.Parameters.volumes:
            vs = float(self.volume_size_re.match(volume['quota']).group('size'))
            if self.min_volume_sizes.get(volume['guest_mp'], 0) > vs:
                volume['quota'] = '{} Gb'.format(self.min_volume_sizes[volume['guest_mp']])
            if volume['guest_mp'] == '/cores' and vs < self.Parameters.mem_requirements * 2:
                volume['quota'] = '{} Gb'.format(self.Parameters.mem_requirements * 2)

    def run_full_checks(self, gencfg, params):
        gencfg.gen_sh('run_checks')

    def get_commit_message(self, gencfg, params):
        return u'[{}] {} from https://sandbox.yandex-team.ru/task/{}/view'.format(
            type(self).__name__, params.commit_message.encode('ascii', 'ignore'), self.id
        )

    def run_subtask_payload(self, gencfg, params):
        self.allocate_groups(gencfg, params)

    @staticmethod
    def list_to_comma_separated(lst):
        return ','.join(map(str, lst))

    @sdk2.header()
    def render_header(self):
        if not self.Context.errors:
            return None

        def render_error(e):
            group = e.get('group', 'Unknown group')
            error_type = e.get('error', 'Unknown error')
            if error_type == 'NOT_ENOUGH_HOSTS':
                error_description = 'Not enough hosts in {geo}: found: {available}, required: {required}'.format(
                    geo=e['geo'], available=e['available_hosts'], required=e['required_hosts']
                )
            elif error_type == 'GROUP_ALREADY_EXISTS':
                error_description = 'Group {group} already exists'.format(group=group)
            else:
                error_description = '{} while allocating {}:{}'.format(error_type, group, e)
            return error_description

        body = [
            '<tr><td>{group}</td><td>{error}</td></tr>'.format(group=e['group'], error=render_error(e)) for e in self.Context.errors
        ]

        return textwrap.dedent("""
            <table bgcolor="red" cols=2>
                <tr>
                    <th>Group</th>
                    <th>Error</th>
                </tr>
                {}
            </table>
        """).strip().format('\n'.join(body))

    def allocate_groups(self, gencfg, params):
        resource = sdk2.ResourceData(SaasGroupsInfo(self, 'SaasGroupsInfo', 'result.json'))
        allocate_groups_command = [
            './utils/saas/allocate_in_saas_cloud.py',
            '--base-group-name',            params.base_name,
            '--tags',                       json.dumps(params.group_tags),
            '--instance-number',            str(params.instance_count),
            '--slot-size-cpu',              str(params.cpu_requirements),
            '--slot-size-mem',              str(params.mem_requirements),
            '--max-instances-per-switch',   str(params.shards_count),
            '--volumes',                    json.dumps(params.volumes),
            '--io-limits',                  json.dumps(params.io_limits),
            '--file',                       str(resource.path)
            ]
        if params.template_group:
            allocate_groups_command.extend(['--template-group', params.template_group])
        if params.owners:
            allocate_groups_command.extend(['--owners', params.owners])
        if params.locations:
            allocate_groups_command.extend(['--geo', ','.join(loc.upper() for loc in params.locations)])
        if params.network_macro:
            allocate_groups_command.extend(['--network-macro', params.network_macro])
        try:
            gencfg.run_process(allocate_groups_command, 'allocate_groups_command')
            resource.ready()
        except subprocess.CalledProcessError:
            with open(str(resource.path)) as allocation_report_file:
                allocation_report = json.load(allocation_report_file)
            self.Context.errors = [i for i in allocation_report if i.get('error', None) is not None]
            resource.ready()
            raise
