# coding: utf-8

# import time
# import logging

# import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from sandbox import sdk2
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.nanny import nanny
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.saas.common.resources import YconfPatcher, SaasRtyserverConfigsBundle


class AssembleRefreshConfigs(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Get configs from svn and release them REFRESH-358 SAAS-4914 """
    """This task also used for images freshness IMAGES-15647       """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 2048
        disk_space = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        arcadia_path = sdk2.parameters.ArcadiaUrl(
            'Arcadia url', required=True, default_value=Arcadia.trunk_url() + '/saas/static_configs/refresh'
        )
        create_oxygen_rt = sdk2.parameters.Bool('create OxygenOptionsRt.cfg', default_value=True)
        create_oxygen_med = sdk2.parameters.Bool('create OxygenOptionsMed.cfg', default_value=True)
        environments = sdk2.parameters.List('Environments', default=['common', 'frozen', 'isolated'])

    def get_arc_path(self, filename):
        return '{}/{}'.format(self.Context.local_configs_path, filename)

    def get_res_path(self, filename):
        return '{}/{}'.format(self.Context.local_resource_path, filename)

    def apply_patch(self, input_file, patch_file, output_file):
        subprocess.check_call(
            ['patch', self.get_arc_path(input_file), '-i', self.get_arc_path(patch_file), '-o', self.get_arc_path(output_file)]
        )

    def apply_yconf_patch(self, input_file, patch_file, output_file):
        """
        Apply yconf patch and add output to resource
        """
        subprocess.check_call(
            [self.Context.yconf_patcher, 'patch', '--ignore-prefix', self.get_arc_path(input_file), self.get_arc_path(patch_file), self.get_res_path(output_file)]
        )

    def validate_opts(self, environments):
        dependent_envs = []
        if 'news_quick' in environments:
            dependent_envs += ['news_quick_frozen', 'news_quick_hamster', 'news_quick_prestable']
        for env in dependent_envs:
            if not env in environments:
                raise ValueError('The list of environments must contain ' + env)

    @staticmethod
    def oxygen_options_filename(env, suffix=""):
        oxygen_options_common = "OxygenOptions{}.cfg".format(suffix)
        if env == 'common':
            return oxygen_options_common
        else:
            return '{}-{}'.format(oxygen_options_common, env)

    def patch_oxygen_options(self):
        patch_rt = 'OxygenOptionsRt.cfg.patch'
        patch_med = 'OxygenOptionsMed.cfg.patch'
        for env in self.Parameters.environments:
            oxygen_options = self.oxygen_options_filename(env)
            oxygen_options_rt = self.oxygen_options_filename(env, suffix="Rt")
            oxygen_options_med = self.oxygen_options_filename(env, suffix="Med")
            if env != 'common':
                patch = '{}.patch'.format(oxygen_options)
                self.apply_patch(self.oxygen_options_filename('common'), patch, oxygen_options)

            self.add_to_resource(oxygen_options)
            if (self.Parameters.create_oxygen_rt):
                self.apply_patch(oxygen_options, patch_rt, oxygen_options_rt)
                self.add_to_resource(oxygen_options_rt)
            if self.Parameters.create_oxygen_med:
                self.apply_patch(oxygen_options, patch_med, oxygen_options_med)
                self.add_to_resource(oxygen_options_med)

    def patch_rtyserver_conf(self):
        for env in self.Parameters.environments:
            if env == 'common':
                self.add_to_resource('rtyserver.conf-{}'.format(env))
            else:
                self.apply_yconf_patch('rtyserver.conf-common', 'rtyserver.conf-{}_patch.json'.format(env), 'rtyserver.conf-{}'.format(env))

    def patch_basesearch_refresh(self):
        for env in self.Parameters.environments:
            if env == 'common':
                self.add_to_resource('basesearch-refresh')
            else:
                self.apply_yconf_patch('basesearch-refresh', 'basesearch-refresh-{}_patch.json'.format(env), 'basesearch-refresh-{}'.format(env))

    def patch_environment(self):
        for env in self.Parameters.environments:
            if env == 'common':
                self.add_to_resource('environment')
            else:
                target_file = 'environment-{}'.format(env)
                self.apply_patch('environment', '{}.patch'.format(target_file), target_file)
                self.add_to_resource(target_file)

    def add_to_resource(self, filename):
        subprocess.check_call(['cp', self.get_arc_path(filename), self.get_res_path(filename)])

    def on_execute(self):
        self.validate_opts(self.Parameters.environments)

        yconf_patcher_resource = YconfPatcher.find(state=ctr.State.READY).first()
        yconf_patcher_resource_data = sdk2.ResourceData(yconf_patcher_resource)

        self.Context.yconf_patcher = str(yconf_patcher_resource_data.path)
        self.Context.local_configs_path = str(sdk2.svn.Arcadia.get_arcadia_src_dir(self.Parameters.arcadia_path))

        resource = SaasRtyserverConfigsBundle(self, self.Parameters.description, 'configs')
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        self.Context.local_resource_path = str(resource_data.path)

        with sdk2.helpers.ProcessLog(self, logger='patch') as pl:
            pl.logger.propagate = 1

            self.patch_oxygen_options()
            self.patch_rtyserver_conf()
            self.patch_basesearch_refresh()
            self.patch_environment()

        resource_data.ready()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)

