# -*- coding: utf-8 -*-

import datetime
import logging
import re

import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr

from sandbox import sdk2
from sandbox.common import config
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny import const
from sandbox.projects.saas.common.resources import FerrymanConfig


class BuildFerrymanConfigs(sdk2.Task, nanny.ReleaseToNannyTask2):

    conf_file_regexp = re.compile(r'.*/(?P<ctype>\w.*)/(?P<filename>config-(?P<service>\w.*)\.pb\.txt)')

    class Parameters(sdk2.Parameters):
        configs_list = sdk2.parameters.List(
            'List of configs (with full svn path)',
            sdk2.parameters.String
        )

    def on_execute(self):
        for config_file in self.Parameters.configs_list:
            match = self.conf_file_regexp.match(str(config_file))
            if match:
                service_name_path = match.group('ctype') + '/' + match.group('service')
                resource = sdk2.ResourceData(FerrymanConfig(self, 'FerrymanConfig', match.group('filename'), service=service_name_path))
                Arcadia.export(Arcadia.trunk_url(path=config_file.lstrip('/trunk/arcadia/')), resource.path)
                resource.ready()
            else:
                just_subpath = False
                for other_config_file in self.Parameters.configs_list:
                    if str(other_config_file) != str(config_file) and str(config_file) in str(other_config_file):
                        just_subpath = True
                if not just_subpath:
                    raise RuntimeError('Invalid name for ferryman config: {}'.format(config_file))

    def get_nanny_release_info(self, additional_parameters, resource):
        now = datetime.datetime.utcnow().isoformat()
        title = 'Ferryman config for ' + resource.service
        spec = {
            'title': title,
            'desc': additional_parameters.get('release_comments'),
            'type': 'SANDBOX_RELEASE',
            'sandboxRelease': {
                'creationTime': now + 'Z',
                'taskType': str(self.type),
                'taskId': str(self.id),
                'releaseAuthor': str(additional_parameters.get('releaser', 'anonymous')),
                'desc': additional_parameters.get('release_comments', ''),
                'title': title,
                'releaseType': additional_parameters['release_status'],
                'resourceTypes': [str(resource.type)],
                'resources': [
                    {
                        'id': str(resource.id),
                        'type': resource.type.name,
                        'description': resource.description,
                        'skynetId': resource.skynet_id,
                        'httpUrl': resource.http_proxy or '',
                        'arch': resource.arch or '',
                        'fileMd5': resource.md5 or '',
                        'releasers': resource.releasers or [],
                    }
                ],
            },
        }
        return {
            'spec': spec,
        }

    def on_release(self, additional_parameters):
        self.mark_released_resources(additional_parameters["release_status"])
        registry = config.Registry()
        if registry.common.installation != ctm.Installation.PRODUCTION:
            logging.info('Current installation is not PRODUCTION but %s, will not send release to Nanny',
                         registry.common.installation)
            return
        resources = [res for res in sdk2.Resource.find(task=self, state=ctr.State.READY).limit(0) if res.type.releasable]
        for resource in resources:
            data = self.get_nanny_release_info(additional_parameters, resource)
            result = self.nanny_client.create_release2(data)['value']
            release_id = result['id']
            if self.Context.nanny_release_requests is ctm.NotExists:
                self.Context.nanny_release_requests = []
            self.Context.nanny_release_requests.append(release_id)
            release_link = const.RELEASE_REQUEST_TEMPLATE.format(
                nanny_api_url=const.NANNY_API_URL,
                release_request_id=release_id,
            )
            self.set_info(
                'Nanny release request <a href="{}">{}</a> was created.'.format(release_link, release_id),
                do_escape=False
            )
